/*
			(c) Copyright 1998-2000 - Tord Jansson
			======================================

		This file is part of the BladeEnc MP3 Encoder, based on
		ISO's reference code for MPEG Layer 3 compression, and might
		contain smaller or larger sections that are directly taken
		from ISO's reference code.

		All changes to the ISO reference code herein are either
		copyrighted by Tord Jansson (tord.jansson@swipnet.se)
		or sublicensed to Tord Jansson by a third party.

	BladeEnc is free software; you can redistribute this file
	and/or modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.

	2000-02-15  Pirandot
*/

#define					OLD_CALC_SCFSI			1	/* There were lots of bugs, but let it on at the moment */
#define					BINARY_SEARCH			1	/* The binary search is faulty implemented, but let it on ... */



#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include "system.h"
#include "l3side.h"
#include "loop.h"
#include "huffman.h"
#include "l3bitstream.h"
#include "reservoir.h"
#include "loop-pvt.h"



double aint (double in);



/*
	Here are MPEG1 Table B.8 and MPEG2 Table B.1
	-- Layer III scalefactor bands.
	Index into this using a method such as:
	idx  = fr_ps->header->sampling_frequency
	       + (fr_ps->header->version * 3)
*/

struct scalefac_struct sfBandIndex[3] =
{
	{ /* Table B.8.b: 44.1 kHz */
		{0,4,8,12,16,20,24,30,36,44,52,62,74,90,110,134,162,196,238,288,342,418,576},
		{0,4,8,12,16,22,30,40,52,66,84,106,136,192}
	},
	{ /* Table B.8.c: 48 kHz */
		{0,4,8,12,16,20,24,30,36,42,50,60,72,88,106,128,156,190,230,276,330,384,576},
		{0,4,8,12,16,22,28,38,50,64,80,100,126,192}
	},
	{ /* Table B.8.a: 32 kHz */
		{0,4,8,12,16,20,24,30,36,44,54,66,82,102,126,156,194,240,296,364,448,550,576},
		{0,4,8,12,16,22,30,42,58,78,104,138,180,192}
	}
};



/*
	The following table is used to implement the scalefactor
	partitioning for MPEG2 as described in section
	2.4.3.2 of the IS. The indexing corresponds to the
	way the tables are presented in the IS:

	[table_number][row_in_table][column of nr_of_sfb]
*/

static unsigned nr_of_sfb_block[6][3][4] =
{
	{
		{ 6, 5, 5, 5},
		{ 9, 9, 9, 9},
		{ 6, 9, 9, 9}
	},
	{
		{6, 5,  7, 3},
		{9, 9, 12, 6},
		{6, 9, 12, 6}
	},
	{
		{11,10, 0, 0},
		{18,18, 0, 0},
		{15,18, 0, 0}
	},
	{
		{ 7, 7, 7, 0},
		{12,12,12, 0},
		{ 6,15,12, 0}
	},
	{
		{ 6, 6, 6, 3},
		{12, 9, 9, 6},
		{ 6,12, 9, 6}
	},
	{
		{ 8, 8, 5, 0},
		{15,12, 9, 0},
		{ 6,18, 9, 0}
	}
};



/* Table B.6: layer3 preemphasis */
int  pretab[21] =
{
	0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 1, 1, 1, 
	1, 2, 2, 3, 3, 3, 2
};



/* This is the scfsi_band table from 2.4.2.7 of the IS */
int scfsi_band_long[5] = { 0, 6, 11, 16, 21 };



int						*scalefac_band_long  = &sfBandIndex[0].l[0];
int						*scalefac_band_short = &sfBandIndex[0].s[0];

int						fInit_iteration_loop;
int						fInit_huffman_read_flag; 

int						tjBitOverflow1;
int						tjBitOverflow2;



void fixStatic_loop( void )
{
	scalefac_band_long  = &sfBandIndex[0].l[0];
	scalefac_band_short = &sfBandIndex[0].s[0];
}





/*  ========================================================================  */
/*              iteration_loop                                                */
/*  ========================================================================  */

void iteration_loop
(
	double					pe[][2],
	double					xr_org[2][2][576],
	III_psy_ratio			*ratio,
	III_side_info_t			*l3_side,
	int						l3_enc[2][2][576],
	int						mean_bits,
	int						stereo,
	double					xr_dec[2][2][576],
	III_scalefac_t			*scalefac,
	frame_params			*fr_ps,
	int						ancillary_pad,
	int						bitsPerFrame
)
{
	III_psy_xmin			l3_xmin;

	int						max_bits;
	int						ch, gr, i, sfb, b, scfsi_band;
	int						mode_gr;

	double					abs_xr[576];

	int						*main_data_begin = &l3_side->main_data_begin;
	layer					*info            = fr_ps->header;


int						start, end;
double					energy_l[SFB_LMAX];
double					sum1, sum2, temp, x;


static double			en_tot[2][2];          /* gr, ch */
static double			en[2][2][SFB_LMAX-1];
static double			xm[2][2][SFB_LMAX-1];
static int				xrAllZero[2][2];

static double			log2;

#define					en_tot_krit				 10.0
#define					en_dif_krit				100.0
#define					en_scfsi_band_krit		 10.0
#define					xm_scfsi_band_krit		 10.0

int						*abs_ix;
D192_3					*abs_xrs;
gr_info					*cod_info;

	log2 = log(2.0);
	l3_side->resvDrain = 0;

	if (!fInit_iteration_loop)
	{
		*main_data_begin = 0;
		fInit_iteration_loop = 1;
	}
	mode_gr = 2;

	scalefac_band_long  = &sfBandIndex[info->sampling_frequency].l[0];
	scalefac_band_short = &sfBandIndex[info->sampling_frequency].s[0];

	/* reading huffman code table */
	if (fInit_huffman_read_flag == 0) 
	{
		read_huffcodetab();
		fInit_huffman_read_flag = 1;
	}


	ResvFrameBegin (fr_ps, l3_side, mean_bits, bitsPerFrame);

	for (gr=0; gr<mode_gr; gr++)
	{
		for (ch=0; ch<stereo; ch++)
		{
			for (i=0; i<576; i++)
				abs_xr[i] = fabs(xr_org[gr][ch][i]);

			abs_ix   = l3_enc[gr][ch];
			abs_xrs  = (D192_3 *) abs_xr;
			cod_info = &l3_side->gr[gr].ch[ch].tt;


/*  ========    gr_deco    ========  */

	if (cod_info->window_switching_flag && (cod_info->block_type == 2))
	{
		if (cod_info->mixed_block_flag)
		{
			cod_info->sfb_lmax = 8;
			cod_info->sfb_smax = 3;
		}
		else
		{
			cod_info->sfb_lmax = 0;  /* No sb*/
			cod_info->sfb_smax = 0;
		}
	}
	else
	{
		cod_info->sfb_lmax = SFB_LMAX-1;
		cod_info->sfb_smax = SFB_SMAX-1;  /* No sb */
	}


/*  ========    calc_xmin and start of quantanf_init    ========  */
/*
	Calculate the allowed distortion for each scalefactor band,
	as determined by the psychoacoustic model.
	xmin(sb) = ratio(sb) * energy(sb) / bandwidth(sb)
*/

	sum1 = 0.0;
	sum2 = 0.0;
	for (sfb=0; sfb<SFB_LMAX; sfb++)
	{
		start = scalefac_band_long[sfb];
		end   = scalefac_band_long[sfb+1];
		temp = 0.0;
		for (i=start; i<end; i++)
		{
			if ((x = abs_xr[i]) != 0.0)
			{
	/*			assert (x < 1.0); */
				temp += x*x;
				sum1 += log(x);
			}
		}
		sum2 += energy_l[sfb] = temp;

		if (sfb < cod_info->sfb_lmax)
			l3_xmin.l[gr][ch][sfb] = ratio->l[gr][ch][sfb] * temp / (end-start);
	}

	for (sfb=cod_info->sfb_smax; sfb<SFB_SMAX-1; sfb++)
	{
		start = scalefac_band_short[sfb];
		end   = scalefac_band_short[sfb+1];
		for (b=0; b<3; b++)
		{
			temp = 0.0;
			for (i=start; i<end; i++)
			{
				temp += (*abs_xrs)[i][b] * (*abs_xrs)[i][b];
			}
			l3_xmin.s[gr][ch][sfb][b] = ratio->s[gr][ch][sfb][b] * temp / (end-start);
		}
	}


/*  ========    calc_scfsi    ========  */

#if OLD_CALC_SCFSI
	/*
		Formerly the maximum of |xr[gr][ch][i]| for i from 0 to 575 was assigned to an int variable xrmax[gr][ch].
		This maximum is always less than 1, so xrmax[gr][ch] was always set to 0!!!
	*/
	xrAllZero[gr][ch] = 1;
#else
	/* the total energy of the granule */
	temp = sum2;
	if (temp == 0.0)
		en_tot[gr][ch] = 0.0;
	else
		en_tot[gr][ch] = log(temp) / log2;

	xrAllZero[gr][ch] = (temp == 0.0);
#endif

	/* None of the granules contains short blocks */
	if (!cod_info->window_switching_flag || (cod_info->block_type != 2))
	{
		for (sfb=0; sfb<SFB_LMAX-1; sfb++)
		{
#if !OLD_CALC_SCFSI
			temp = energy_l[sfb];
			if (temp == 0.0)
				en[gr][ch][sfb] = 0.0;
			else
				en[gr][ch][sfb] = log(temp) / log2;
#endif
			if (l3_xmin.l[gr][ch][sfb] == 0.0)
				xm[gr][ch][sfb] = 0.0;
			else
				xm[gr][ch][sfb] = log(l3_xmin.l[gr][ch][sfb]) / log2;
		}

		if (gr == 1)
		{
			for (scfsi_band=0; scfsi_band<4; scfsi_band++)
				l3_side->scfsi[ch][scfsi_band] = 0;

			/* The spectral values are not all zero */
			if (xrAllZero[0][ch] || xrAllZero[1][ch])  /* always, if OLD_CALC_SCFSI */
				goto calc_scfsi_break;

			if (fabs(en_tot[0][ch]-en_tot[1][ch]) >= en_tot_krit)
				goto calc_scfsi_break;

			temp = 0.0;
			for (sfb=0; sfb<SFB_LMAX-1; sfb++) 
				temp += fabs(en[0][ch][sfb]-en[1][ch][sfb]);
			if (temp >= en_dif_krit)
				goto calc_scfsi_break;

			for (scfsi_band=0; scfsi_band<4; scfsi_band++)
			{
				sum1 = sum2 = 0,0;

				start = scfsi_band_long[scfsi_band];
				end   = scfsi_band_long[scfsi_band+1];
				for (sfb=start; sfb<end; sfb++)
				{ 
					sum1 += fabs(en[0][ch][sfb]-en[1][ch][sfb]);
					sum2 += fabs(xm[0][ch][sfb]-xm[1][ch][sfb]);
				}

				if (sum1 < en_scfsi_band_krit && sum2 < xm_scfsi_band_krit)
					l3_side->scfsi[ch][scfsi_band] = 1;
			}
		}
	}
	calc_scfsi_break:


			/* calculation of number of available bit( per granule ) */
			max_bits = ResvMaxBits (fr_ps, l3_side, &pe[gr][ch], mean_bits);

			/* reset of iteration variables */

			for (sfb=0; sfb<SFB_LMAX-1; sfb++)
				scalefac->l[gr][ch][sfb] = 0;
			for (sfb=0; sfb<SFB_SMAX-1; sfb++)
				for (b=0; b<3; b++)
					scalefac->s[gr][ch][sfb][b] = 0;

			for (scfsi_band=0; scfsi_band<4; scfsi_band++)
				cod_info->slen[scfsi_band] = 0;

			cod_info->sfb_partition_table = &nr_of_sfb_block[0][0][0];
			cod_info->part2_3_length      = 0;
			cod_info->big_values          = 0;
			cod_info->count1              = 0;
			cod_info->scalefac_compress   = 0;
			cod_info->table_select[0]     = 0;
			cod_info->table_select[1]     = 0;
			cod_info->table_select[2]     = 0;
			cod_info->subblock_gain[0]    = 0;
			cod_info->subblock_gain[1]    = 0;
			cod_info->subblock_gain[2]    = 0;
			cod_info->region0_count       = 0;
			cod_info->region1_count       = 0;
			cod_info->part2_length        = 0;
			cod_info->preflag             = 0;
			cod_info->scalefac_scale      = 0;
			cod_info->quantizerStepSize   = 0.0;
			cod_info->count1table_select  = 0;

			/* all spectral values zero ? */
			if (sum2 != 0.0)
			{

/*  ========    quantanf_init (remaining)    ========  */

#define system_const			   8.0
#define minlimit				-100.0

	temp = nint (system_const * (sum1/288.0 - log(sum2/576.0)));
	if (temp < minlimit)
		temp = minlimit;
	/*
		SS 19-12-96. Starting value of
		global_gain or quantizerStepSize 
		has to be reduced for iteration_loop
	*/
	temp -= 70.0;


				cod_info->quantizerStepSize = temp;


				cod_info->part2_3_length = outer_loop (abs_xr, max_bits, &l3_xmin, abs_ix, fr_ps, scalefac, gr, ch, l3_side);
			}

			ResvAdjust (fr_ps, cod_info, l3_side, mean_bits);

			cod_info->global_gain = nint (cod_info->quantizerStepSize + 210.0);
/*			assert (cod_info->global_gain < 256); */
		}	/* for ch */
	}	/* for gr */

	ResvFrameEnd (fr_ps, l3_side, mean_bits);
}





/*  ========================================================================  */
/*              outer_loop                                                    */
/*  ========================================================================  */
/*
	The outer iteration loop controls the masking conditions
	of all scalefactorbands. It computes the best scalefac and
	global gain. This module calls the inner iteration loop
*/

int outer_loop
(
	double					abs_xr[576],	/* magnitudes of the spectral values */
	int						max_bits,
	III_psy_xmin			*l3_xmin,		/* the allowed distortion of the scalefactor */
	int						abs_ix[576],	/* vector of quantized values ix(0..575) */
	frame_params			*fr_ps,
	III_scalefac_t			*scalefac,		/* scalefactors */
	int						gr,
	int						ch,
	III_side_info_t			*l3_side
)
{
	double					xfsf[4][CBLIMIT];
	int						scalesave_l[SFB_LMAX-1];
	int						scalesave_s[SFB_SMAX-1][3];
	int						bits, huff_bits, save_preflag, save_compress;
	int						sfb, b, over, iteration;

	gr_info					*cod_info = &l3_side->gr[gr].ch[ch].tt;

	iteration = 0;
	do 
	{
		iteration++;

		cod_info->part2_length = part2_length (scalefac, fr_ps, gr, ch, l3_side);
		huff_bits = max_bits - cod_info->part2_length;

#ifdef BINARY_SEARCH   /* This function originally contains a bug!!!! */
		if (iteration == 1)
		{
			bin_search_StepSize (max_bits, cod_info->quantizerStepSize, abs_ix, abs_xr, cod_info);  /* speeds things up a bit */
		}
#endif

		bits = inner_loop (abs_xr, abs_ix, huff_bits, cod_info);

		calc_noise (abs_xr, abs_ix, cod_info, xfsf);  /* distortion calculation */

		for (sfb=0; sfb<SFB_LMAX-1; sfb++)  /* save scaling factors */
			scalesave_l[sfb] = scalefac->l[gr][ch][sfb];

		for (sfb=0; sfb<SFB_SMAX-1; sfb++)
			for (b=0; b<3; b++)
				scalesave_s[sfb][b] = scalefac->s[gr][ch][sfb][b];

		save_preflag  = cod_info->preflag;
		save_compress = cod_info->scalefac_compress;

		preemphasis (abs_xr, xfsf, l3_xmin, gr, ch, l3_side);
		over = amp_scalefac_bands (abs_xr, xfsf, l3_xmin, l3_side, scalefac, gr, ch, iteration);

		if (loop_break (scalefac, cod_info, gr, ch))
			break;
		if (scale_bitcount (scalefac, cod_info, gr, ch))
			break;
	}
	while (over > 0);

	cod_info->preflag = save_preflag;
	cod_info->scalefac_compress = save_compress;

	for (sfb=0; sfb<SFB_LMAX-1; sfb++)
		scalefac->l[gr][ch][sfb] = scalesave_l[sfb];    

	for (sfb=0; sfb<SFB_SMAX-1; sfb++)
		for (b=0; b <3; b++)
			scalefac->s[gr][ch][sfb][b] = scalesave_s[sfb][b];

	cod_info->part2_length   = part2_length (scalefac, fr_ps, gr, ch, l3_side);
	cod_info->part2_3_length = cod_info->part2_length + bits;

	return cod_info->part2_3_length;
}





/*  ========================================================================  */
/*              inner_loop                                                    */
/*  ========================================================================  */
/*
	The code selects the best quantizerStepSize for a
	particular set of scalefacs
*/

int inner_loop
(
	double					*abs_xr,
	int						*abs_ix,
	int						max_bits,
	gr_info					*cod_info
)
{
	int bits;

	cod_info->quantizerStepSize -= 1.0;
	do
	{
		do
		{
			cod_info->quantizerStepSize += 1.0;
			tjBitOverflow2 = FALSE;
			quantize_tj (abs_xr, abs_ix, cod_info);
		}
		while (tjBitOverflow2);  /* within table range? */

		bits = count_bits (abs_ix, cod_info);
	}
	while (bits > max_bits);

	return bits;
}





/***************************************************************************/ 
/*        part2_length                                                     */ 
/***************************************************************************/ 

/* calculates the number of bits needed to encode the scalefacs in the     */
/* main data block                                                         */

int part2_length( III_scalefac_t *scalefac, frame_params *fr_ps,
	      int gr, int ch, III_side_info_t *si )
{
  int slen1, slen2, bits;
  gr_info *gi = &si->gr[gr].ch[ch].tt;


	static int slen1_tab[16] = { 0, 0, 0, 0, 3, 1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4 };
	static int slen2_tab[16] = { 0, 1, 2, 3, 0, 1, 2, 3, 1, 2, 3, 1, 2, 3, 2, 3 };

  bits = 0;


	slen1 = slen1_tab[ gi->scalefac_compress ];
	slen2 = slen2_tab[ gi->scalefac_compress ];

	if ( (gi->window_switching_flag == 1) && (gi->block_type == 2) )
	{
	  if ( gi->mixed_block_flag )
	  {
			bits += (8 * slen1) + (9 * slen1) + (18 * slen2);
	  }
	  else
	  {
			bits += (18 * slen1) + (18 * slen2);
	  }
	}
	else
	{
	  if ( (gr == 0) || (si->scfsi[ch][0] == 0) )
			bits += (6 * slen1);

	  if ( (gr == 0) || (si->scfsi[ch][1] == 0) )
			bits += (5 * slen1);

	  if ( (gr == 0) || (si->scfsi[ch][2] == 0) )
			bits += (5 * slen2);

	  if ( (gr == 0) || (si->scfsi[ch][3] == 0) )
			bits += (5 * slen2);
	}
    return bits;
}





/*************************************************************************/
/*            scale_bitcount                                             */
/*************************************************************************/


/* Also calculates the number of bits necessary to code the scalefactors. */

int scale_bitcount( III_scalefac_t *scalefac, gr_info *cod_info,
		int gr, int ch )
{
    int i, k, sfb, max_slen1 = 0, max_slen2 = 0, /*a, b, */ ep = 2;


    static int slen1[16] = { 0, 0, 0, 0, 3, 1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4 };
    static int slen2[16] = { 0, 1, 2, 3, 0, 1, 2, 3, 1, 2, 3, 1, 2, 3, 2, 3 };
    static int pow2[5]   = { 1, 2, 4, 8, 16 };

    if ( cod_info->window_switching_flag != 0 && cod_info->block_type == 2 )
    {
        if ( cod_info->mixed_block_flag == 0 ) 
        {
            /* a = 18; b = 18;  */
            for ( i = 0; i < 3; i++ )
            {
                for ( sfb = 0; sfb < 6; sfb++ )
                    if ( scalefac->s[gr][ch][sfb][i] > max_slen1 )
                        max_slen1 = scalefac->s[gr][ch][sfb][i];
                for (sfb = 6; sfb < 12; sfb++ )
                    if ( scalefac->s[gr][ch][sfb][i] > max_slen2 )
                        max_slen2 = scalefac->s[gr][ch][sfb][i];
            }
        }
        else
        {/* mixed_block_flag = 1 */
            /* a = 17; b = 18;  */
            for ( sfb = 0; sfb < 8; sfb++ )
                if ( scalefac->l[gr][ch][sfb] > max_slen1 )
                    max_slen1 = scalefac->l[gr][ch][sfb];
            for ( i = 0; i < 3; i++ )
            {
                for ( sfb = 3; sfb < 6; sfb++ )
                    if ( scalefac->s[gr][ch][sfb][i] > max_slen1 )
                        max_slen1 = scalefac->s[gr][ch][sfb][i];
                for ( sfb = 6; sfb < 12; sfb++ )
                    if ( scalefac->s[gr][ch][sfb][i] > max_slen2 )
                        max_slen2 = scalefac->s[gr][ch][sfb][i];
            }
        }

    }
    else
    { /* block_type == 1,2,or 3 */
        /* a = 11; b = 10;   */
        for ( sfb = 0; sfb < 11; sfb++ )
            if ( scalefac->l[gr][ch][sfb] > max_slen1 )
                max_slen1 = scalefac->l[gr][ch][sfb];
        for ( sfb = 11; sfb < 21; sfb++ )
            if ( scalefac->l[gr][ch][sfb] > max_slen2 )
                max_slen2 = scalefac->l[gr][ch][sfb];
    }

    for ( k = 0; k < 16; k++ )
    {
        if ( (max_slen1 < pow2[slen1[k]]) && (max_slen2 < pow2[slen2[k]]) )
        { 
            ep = 0;
            break;
        } 
    }

    if ( ep == 0 )
        cod_info->scalefac_compress = k;
    return ep;
}





/*************************************************************************/
/*            calc_noise                                                 */
/*************************************************************************/

double					noisePowTab[8191+15];

void genNoisePowTab (void)
{
	int	i;

	for (i=0; i<8191+15; i++)
		noisePowTab[i] = pow (i, 4.0/3.0);
}



/*   Function: Calculate the distortion introduced by the quantization   */
/*   in each scale factor band.                                          */

void calc_noise( double xr[576], int ix[576], gr_info *cod_info,
	    double xfsf[4][CBLIMIT] )
{
  int start, end, l, i;
	unsigned int sfb;
  double sum,step,bw;

  D192_3 *xr_s;
  I192_3 *ix_s;

  xr_s = (D192_3 *) xr;
  ix_s = (I192_3 *) ix;

  step = pow( 2.0, (cod_info->quantizerStepSize) * 0.25 );
  for ( sfb = 0; sfb < cod_info->sfb_lmax; sfb++ )
  {
    start = scalefac_band_long[ sfb ];
    end   = scalefac_band_long[ sfb+1 ];
		bw = end - start;

    for ( sum = 0.0, l = start; l < end; l++ )
    {
      double temp;
      temp = xr[l] - noisePowTab[ix[l]] * step;
      sum += temp * temp; 
    }
    xfsf[0][sfb] = sum / bw;
  }

  for ( i = 0; i < 3; i++ )
  {
    for ( sfb = cod_info->sfb_smax; sfb < 12; sfb++ )
    {
      start = scalefac_band_short[ sfb ];
      end   = scalefac_band_short[ sfb+1 ];
			bw = end - start;
      
      for ( sum = 0.0, l = start; l < end; l++ )
      {
        double temp;
        temp = (*xr_s)[l][i] - noisePowTab[(*ix_s)[l][i]] * step;
        sum += temp * temp;
      }       
      xfsf[i+1][sfb] = sum / bw;
    }
  }
}





/*************************************************************************/
/*            loop_break                                                 */
/*************************************************************************/

/*  Function: Returns zero if there is a scalefac which has not been
    amplified. Otherwise it returns one. 
*/

int loop_break( III_scalefac_t *scalefac, gr_info *cod_info,
	    int gr, int ch )
{
    uint i, sfb;

    for ( sfb = 0; sfb < cod_info->sfb_lmax; sfb++ )
        if ( scalefac->l[gr][ch][sfb] == 0 )
            return 0;

    for ( sfb = cod_info->sfb_smax; sfb < 12; sfb++ )
        for ( i = 0; i < 3; i++ )
            if ( scalefac->s[gr][ch][sfb][i] == 0 )
                return 0;
    return 1;
}





/*************************************************************************/
/*            preemphasis                                                */
/*************************************************************************/

/*
  See ISO 11172-3  section  C.1.5.4.3.4
*/

void preemphasis( double xr[576], double xfsf[4][CBLIMIT],
	     III_psy_xmin  *l3_xmin,
	     int gr, int ch, III_side_info_t *l3_side )
{
  int i, start, end, scfsi_band, over;
	uint sfb;
  double ifqstep;
  gr_info *cod_info = &l3_side->gr[gr].ch[ch].tt;
	double	mulval;

  if ( gr == 1 )
  {
		/*
			If the second granule is being coded and scfsi is active in
			at least one scfsi_band, the preemphasis in the second granule
			is set equal to the setting in the first granule
		*/
		for ( scfsi_band = 0; scfsi_band < 4; scfsi_band++ )
			if ( l3_side->scfsi[ch][scfsi_band] )
			{
				cod_info->preflag = l3_side->gr[0].ch[ch].tt.preflag;
				return;
			}

  }

  /*
    Preemphasis is switched on if in all the upper four scalefactor
    bands the actual distortion exceeds the threshold after the
    first call of the inner loop
  */
  if ( cod_info->block_type != 2 && cod_info->preflag == 0 )
  {	
		over = 0;
		for ( sfb = 17; sfb < 21; sfb++ )
			if ( xfsf[0][sfb] > l3_xmin->l[gr][ch][sfb] )
				over++;

		if (over == 4 )
		{
			cod_info->preflag = 1;
			ifqstep = ( cod_info->scalefac_scale == 0 ) ? sqrt(2.)
	:		pow( 2.0, (0.5 * (1.0 + (double) cod_info->scalefac_scale)) );

			for ( sfb = 0; sfb < cod_info->sfb_lmax; sfb++ )
			{
				l3_xmin->l[gr][ch][sfb] *= pow( ifqstep, 2.0 * (double) pretab[sfb] );
				start = scalefac_band_long[ sfb ];
				end   = scalefac_band_long[ sfb+1 ];

				mulval = pow( ifqstep, (double) pretab[sfb] );
				for( i = start; i < end; i++ )
					xr[i] *= mulval;
			}
		}
  }
}





/*************************************************************************/
/*            amp_scalefac_bands                                         */
/*************************************************************************/

/* 
  Amplify the scalefactor bands that violate the masking threshold.
  See ISO 11172-3 Section C.1.5.4.3.5
*/

int amp_scalefac_bands( double xr[576], double xfsf[4][CBLIMIT],
		    III_psy_xmin *l3_xmin, III_side_info_t *l3_side,
		    III_scalefac_t *scalefac,
		    int gr, int ch, int iteration )
{
  int start, end, l, i, scfsi_band, over = 0;
	uint sfb;
  double ifqstep, ifqstep2;
  D192_3 *xr_s;
  gr_info *cod_info, *gr0;
  int copySF, preventSF;
  cod_info = &l3_side->gr[gr].ch[ch].tt;
  gr0      = &l3_side->gr[0].ch[ch].tt;


  xr_s = (D192_3 *) xr;
  copySF = 0;
  preventSF = 0;

  if ( cod_info->scalefac_scale == 0 )
		ifqstep = 1.414213562;							/* sqrt( 2.0 ); */
  else
		ifqstep = pow( 2.0, 0.5 * (1.0 + (double) cod_info->scalefac_scale) );

  if ( gr == 1 )
  {
		for ( scfsi_band = 0; scfsi_band < 4; scfsi_band++ )
	    if ( l3_side->scfsi[ch][scfsi_band] )
	    {
				if ( gr0->scalefac_scale == 0 )

					ifqstep = 1.414213562;				/* sqrt( 2.0 ); */
				else
					ifqstep = pow( 2.0, 0.5 * (1.0 + (double) gr0->scalefac_scale) );

				if ( iteration == 1 )
					copySF = 1;
				else
			    preventSF = 1;
				break;
	    }
	
  }

  ifqstep2 = ifqstep * ifqstep;
  scfsi_band = 0;
    
  for ( sfb = 0; sfb < cod_info->sfb_lmax; sfb++ )
  {
		if ( copySF || preventSF )
		{
	    if ( (int) sfb == scfsi_band_long[scfsi_band + 1] )
				scfsi_band += 1;
	    if ( l3_side->scfsi[ch][scfsi_band] )
	    {
				if ( copySF )
		    scalefac->l[gr][ch][sfb] = scalefac->l[0][ch][sfb];
				continue;
	    }
		}	    
		if ( xfsf[0][sfb] > l3_xmin->l[gr][ch][sfb] )
		{
	    over++;
	    l3_xmin->l[gr][ch][sfb] *= ifqstep2;
	    scalefac->l[gr][ch][sfb]++;
	    start = scalefac_band_long[sfb];
	    end   = scalefac_band_long[sfb+1];
	    for ( l = start; l < end; l++ )
				xr[l] *= ifqstep;
		}
  }

  for ( i = 0; i < 3; i++ )
    for ( sfb = cod_info->sfb_smax; sfb < 12; sfb++ )
      if ( xfsf[i+1][sfb] > l3_xmin->s[gr][ch][sfb][i] )
      {
        over++;
        l3_xmin->s[gr][ch][sfb][i] *= ifqstep2;
        scalefac->s[gr][ch][sfb][i]++;
        start = scalefac_band_short[sfb];
        end   = scalefac_band_short[sfb+1];
        for ( l = start; l < end; l++ )
            (*xr_s)[l][i] *= ifqstep;
      }
  return over;
}





/*  ========================================================================  */
/*              quantize                                                      */
/*  ========================================================================  */
/*
	Quantization of the vector xr ( -> ix)
*/



extern	unsigned short			bladeTablesh[];
extern	FLOAT					bladeTablefloat[];



/*____ bladTabValue() _________________________________________________________*/

int INLINE bladTabValue (FLOAT in)
{
	int						in2 = *((int *) &in);
	int						in1;
	int						retVal;

	if (in2 >= 0x48213ba6)
	{
		retVal = nint (pow(in, 0.75) - 0.0946);
		if (retVal > 8192)
		{
			tjBitOverflow1 = TRUE;
			if (retVal > 8191 + 14)
				tjBitOverflow2 = TRUE;
		}		
		return retVal;
	}

	in1 = in2 - 0x3effff7a;
	if (in1 < 0)
		return 0;

	retVal = bladeTablesh[in1>>10];
	if (retVal & 0x8000)
	{
		retVal &= 0x7fff;
		if (in2 < ((int *)bladeTablefloat)[retVal])
			retVal--;
	}
	return retVal;
}



/*____ quantize_tj() __________________________________________________________*/

void quantize_tj
(
	double					abs_xr[576],
	int						abs_ix[576],
	gr_info					*cod_info
)
{
	int						i, b, l_end, s_start;
	double					expo, mulVal;
	float					w, x, y, z;

	D192_3					*abs_xr_s = (D192_3 *) abs_xr;
	I192_3					*abs_ix_s = (I192_3 *) abs_ix;

	expo   = - cod_info->quantizerStepSize * 0.25;
	mulVal = pow (2.0, expo);

	if (cod_info->window_switching_flag && (cod_info->block_type == 2))
	{
		if ( cod_info->mixed_block_flag)
		{
			l_end   = 18 * 2;
			s_start =  6 * 2;
		}
		else
		{
			l_end   = 0;
			s_start = 0;
		}
		memset (abs_ix+l_end, 0, (576-l_end) * sizeof(abs_ix[0]));
	}
	else
	{
		l_end   = 576;
		s_start = 192;
	}

	for (i=0; i<l_end; i+=4)
	{
/*		abs_ix[i] = nint (pow (abs_xr[i] * mulVal, 0.75) - 0.0946); */
		w = abs_xr[i  ] * mulVal;
		x = abs_xr[i+1] * mulVal;
		y = abs_xr[i+2] * mulVal;
		z = abs_xr[i+3] * mulVal;
		abs_ix[i  ] = bladTabValue (w);
		abs_ix[i+1] = bladTabValue (x);
		abs_ix[i+2] = bladTabValue (y);
		abs_ix[i+3] = bladTabValue (z);
	}


	if (s_start < 192)
	{
		for (b=0; b<3; b++)
		{
			mulVal = pow (2.0, expo - (double) (cod_info->subblock_gain[b] * 2));
			for (i=s_start; i<192; i++)
			{
/*				(*abs_ix_s)[i][b] = nint (pow ((*abs_xr_s)[i][b]) * mulVal, 0.75) - 0.0946); */
				(*abs_ix_s)[i][b] = bladTabValue ((*abs_xr_s)[i][b] * mulVal);
			}
		}
	}
}





/*  ========================================================================  */
/*              ix_max_tj                                                     */
/*  ========================================================================  */
/*
	Calculate the maximum of abs_ix from begin to end
*/


int ix_max_tj
(
	int						abs_ix[576],
	unsigned int			begin,
	unsigned int			end
)
{
	int						i;
	int						max = 0;

	for (i=begin; i<end; i++)
		if (abs_ix[i] > max)
			max = abs_ix[i];

	return max;
}





/*  ========================================================================  */
/*              count_bits                                                    */
/*  ========================================================================  */

struct
{
	unsigned region0_count;
	unsigned region1_count;
} subdv_table[ 23 ] =
{
	{0, 0}, /*  0 bands */
	{0, 0}, /*  1 bands */
	{0, 0}, /*  2 bands */
	{0, 0}, /*  3 bands */
	{0, 0}, /*  4 bands */
	{0, 1}, /*  5 bands */
	{1, 1}, /*  6 bands */
	{1, 1}, /*  7 bands */
	{1, 2}, /*  8 bands */
	{2, 2}, /*  9 bands */
	{2, 3}, /* 10 bands */
	{2, 3}, /* 11 bands */
	{3, 4}, /* 12 bands */
	{3, 4}, /* 13 bands */
	{3, 4}, /* 14 bands */
	{4, 5}, /* 15 bands */
	{4, 5}, /* 16 bands */
	{4, 6}, /* 17 bands */
	{5, 6}, /* 18 bands */
	{5, 6}, /* 19 bands */
	{5, 7}, /* 20 bands */
	{6, 7}, /* 21 bands */
	{6, 7}, /* 22 bands */
};



/*
	Calculation of rzero, count1, big_values
	(Partitions ix into big values, quadruples and zeros).

	Determines the number of bits to encode the quadruples.

	Presumable subdivides the bigvalue region which will
	use separate Huffman tables.

	Select huffman code tables for bigvalues regions

	Count the number of bits necessary to code the bigvalues region.
*/

int count_bits
(
	int						abs_ix[576],
	gr_info					*cod_info
)
{
	int						zero_region = 576;
	int						bigv_region = 576;

	int						bits = 0;
	int						sum0 = 0;
	int						sum1 = 0;

	if (!cod_info->window_switching_flag || (cod_info->block_type != 2))
	{
		int						p;

		for (; zero_region>1; zero_region-= 2)
			     if (abs_ix[zero_region-1])  break;
			else if (abs_ix[zero_region-2])  break;

		for (bigv_region=zero_region; bigv_region>3; bigv_region-= 4)
		{
			     if (abs_ix[bigv_region-1] > 1)  break;
			else if (abs_ix[bigv_region-2] > 1)  break;
			else if (abs_ix[bigv_region-3] > 1)  break;
			else if (abs_ix[bigv_region-4] > 1)  break;

			p = 0;
			if (abs_ix[bigv_region-1])  bits++, p |= 8;
			if (abs_ix[bigv_region-2])  bits++, p |= 4;
			if (abs_ix[bigv_region-3])  bits++, p |= 2;
			if (abs_ix[bigv_region-4])  bits++, p |= 1;

			sum0 += ht[32].hlen[p];
			sum1 += ht[33].hlen[p];
		}
	}
	cod_info->count1     = (zero_region-bigv_region) / 4;
	cod_info->big_values =              bigv_region  / 2;

	if (sum0 < sum1)
	{
		bits += sum0;
		cod_info->count1table_select = 0;
	}
	else
	{
		bits += sum1;
		cod_info->count1table_select = 1;
	}

	if (bigv_region)
	{
		if (cod_info->window_switching_flag)
		{
			if (cod_info->mixed_block_flag || (cod_info->block_type != 2))
			{
				cod_info->region0_count =  7;
				cod_info->region1_count = 13;
				cod_info->address1 = scalefac_band_long[cod_info->region0_count+1];
			}
			else
			{ 
				cod_info->region0_count =  8;
				cod_info->region1_count = 36;
				cod_info->address1 = 36;
			}
			cod_info->address2 = bigv_region;
			cod_info->address3 = 0;
		}
		else
		{	/* long blocks */

/*  ========    This area replaced my favourite joke in original subdivide()    ========  */

			int						index0, index1;
			int						scfb_anz = 0;

			/* Calculate scfb_anz */
			while (scalefac_band_long[scfb_anz] < bigv_region)
				scfb_anz++;
/*			assert (scfb_anz < 23); */

			index0 = (cod_info->region0_count = subdv_table[scfb_anz].region0_count) + 1;
			index1 = (cod_info->region1_count = subdv_table[scfb_anz].region1_count) + 1;

			cod_info->address1 = scalefac_band_long[index0];
			cod_info->address2 = scalefac_band_long[index0 + index1];
			cod_info->address3 = bigv_region;

/*  ======================================================================================  */

		}
	}
	else
	{	/* no big_values region */
		cod_info->region0_count = 0;
		cod_info->region1_count = 0;
	}


	cod_info->table_select[0] = 0;
	cod_info->table_select[1] = 0;
	cod_info->table_select[2] = 0;

	if (cod_info->window_switching_flag && (cod_info->block_type == 2))
	{
		/*
			Within each scalefactor band, data is given for successive
			time windows, beginning with window 0 and ending with window 2.
			Within each window, the quantized values are then arranged in
			order of increasing frequency...
		*/
		int						sfb, window, line, start, end;
		unsigned int			max1, max2, *pmax, tMax;
		int						x, y;

		I192_3					*ix_s = (I192_3 *) abs_ix;

		max1 = max2 = 0;

		for (sfb=0; sfb<SFB_SMAX; sfb++)
		{
			start = scalefac_band_short[sfb];
			end   = scalefac_band_short[sfb+1];

			if (start < 12)
				pmax = &max1;
			else
				pmax = &max2;

			tMax = *pmax;

			for (window=0; window<3; window++)
			{
				for (line=start; line<end; line+=2)
				{
					x = (*ix_s)[line  ][window];
					y = (*ix_s)[line+1][window];

					if (x > tMax)  tMax = x;
					if (y > tMax)  tMax = y;
				}
			}

			*pmax = tMax;
		}
		cod_info->table_select[0] = choose_table(max1);
		cod_info->table_select[1] = choose_table(max2);

		if (cod_info->mixed_block_flag)
		{
			if (cod_info->table_select[0])
				bits += count_bit_tj (abs_ix, 0, cod_info->address1, cod_info->table_select[0]);
			sfb = 2;
		}
		else
			sfb = 0;

		for ( ; sfb<SFB_SMAX; sfb++)
		{
			unsigned				tableindex;
			unsigned				idx;
			struct huffcodetab		*h;

			start = scalefac_band_short[sfb];
			end   = scalefac_band_short[sfb+1];

			if (start < 12)
				tableindex = cod_info->table_select[0];
			else
				tableindex = cod_info->table_select[1];
/*			assert (tableindex < 32); */

			if (tableindex != 0)
			{
				h = &(ht[tableindex]);

				for (window=0; window<3; window++)
				{
					for (line=start; line<end; line+=2)
					{
						int x = (*ix_s)[line  ][window];
						int y = (*ix_s)[line+1][window];

						if (tableindex > 15)
						{	/* ESC-table is used */
							if (x > 14)
							{
								x = 15;
								bits += h->linbits;
							}
							if (y > 14)
							{
								y = 15;
								bits += h->linbits;
							}
						}

						idx = (x * h->ylen) + y;
						bits += h->hlen[ idx ];

						if (x)  bits++;
						if (y)  bits++;
					}
				}
			}
		}
	}
	else
	{
		if (cod_info->address1 > 0)
			if ((cod_info->table_select[0] = new_choose_table_tj (abs_ix, 0, cod_info->address1)) > 0)
				bits += count_bit_tj (abs_ix, 0, cod_info->address1, cod_info->table_select[0]);  /* region0 */

		if (cod_info->address2 > cod_info->address1)
			if ((cod_info->table_select[1] = new_choose_table_tj (abs_ix, cod_info->address1, cod_info->address2)) > 0)
				bits += count_bit_tj (abs_ix, cod_info->address1, cod_info->address2, cod_info->table_select[1]);  /* region1 */

		if (cod_info->big_values * 2 > cod_info->address2)
			if ((cod_info->table_select[2] = new_choose_table_tj (abs_ix, cod_info->address2, cod_info->big_values*2)) > 0)
				bits += count_bit_tj (abs_ix, cod_info->address2, cod_info->address3, cod_info->table_select[2]);  /* region2 */
	}

	return bits;
}





/*  ========================================================================  */
/*              new_choose table                                              */
/*  ========================================================================  */
/*
	Choose the Huffman table that will encode ix[begin..end] with
	the fewest bits.

	Note: This code contains knowledge about the sizes and characteristics
	of the Huffman tables as defined in the IS (Table B.7), and will not work
	with any arbitrary tables.
*/

int new_choose_table_tj
(
	int						abs_ix[576],
	unsigned int			begin,
	unsigned int			end
)
{
	uint					max, choice0, choice1;
	int						sum0, sum1;

	max = ix_max_tj (abs_ix, begin, end);

	if (max == 0)
		return 0;

	choice0 = 0;
	choice1 = 0;

	if (max < 15)
	{
		choice0 = 1;  /* not 0 -- ht[0].xlen == 0 */
		while (ht[choice0].xlen <= max)
			choice0++;

		sum0 = count_bit_tj (abs_ix, begin, end, choice0);

		switch ( choice0 )
		{
			case  2:  sum1 = count_bit_tj (abs_ix, begin, end,  3); if (sum1 <= sum0) choice0 =  3; break;
			case  5:  sum1 = count_bit_tj (abs_ix, begin, end,  6); if (sum1 <= sum0) choice0 =  6; break;

			case  7:  sum1 = count_bit_tj (abs_ix, begin, end,  8); if (sum1 <= sum0)  {choice0 =  8; sum0 = sum1;}
					  sum1 = count_bit_tj (abs_ix, begin, end,  9); if (sum1 <= sum0) choice0 =  9; break;

			case 10:  sum1 = count_bit_tj (abs_ix, begin, end, 11); if (sum1 <= sum0)  {choice0 = 11; sum0 = sum1;}
					  sum1 = count_bit_tj (abs_ix, begin, end, 12); if (sum1 <= sum0) choice0 = 12; break;

			case 13:  sum1 = count_bit_tj (abs_ix, begin, end, 15); if (sum1 <= sum0) choice0 = 15; break;

			default:  break;
		}
	}
	else
	{
		/* try tables with linbits */
		max -= 15;

		choice0 = 15;
		while (ht[choice0].linmax < max)
			choice0++;

		choice1 = 24;
		while (ht[choice1].linmax < max)
			choice1++;

		sum0 = count_bit_tj (abs_ix, begin, end, choice0);
		sum1 = count_bit_tj (abs_ix, begin, end, choice1);
		if (sum1 < sum0)
			choice0 = choice1;
	}

	return choice0;
}





/*  ========================================================================  */
/*              choose table                                                  */
/*  ========================================================================  */

int choose_table (unsigned int max)
{
	unsigned int			choice0;

	if (max == 0)
		return 0;

	if ( max < 15 )
	{
		choice0 = 1;  /* not 0 -- ht[0].xlen == 0 */
		while (ht[choice0].xlen <= max)
			choice0++;
	}
	else
	{	
		max -= 15;

		choice0 = 15;
		while (ht[choice0].linmax < max)
			choice0++;
	}

	return choice0;
}





/*  ========================================================================  */
/*              count_bit                                                     */
/*  ========================================================================  */
/*
	Count the number of bits necessary to code the subregion. 
*/

int count_bit_tj
(
	int						abs_ix[576],
	unsigned int			start,
	unsigned int			end,
	unsigned int			table
)
{
	int						i, sum;
	unsigned				idx;
	struct huffcodetab		*h;
	int						x, y;

	if (table == 0)
		return 0;

	h = &(ht[table]);

	sum = 0;
	for (i=start; i<end; i+=2)
	{
		x = abs_ix[i];
		y = abs_ix[i+1];

		if (table > 15)
		{
			if (x > 14)
			{
				x = 15;
				sum += h->linbits;
			}
			if (y > 14)
			{
				y = 15;
				sum += h->linbits;
			}
		}

		idx = (x * h->ylen) + y;
		sum += h->hlen[idx];

		if (x)  sum++;
		if (y)  sum++;
	}

	return sum;
}





#ifndef HAVE_NINT
int nint (double in)
{
	int						temp;

	if (in < 0)
		temp = (int)(in - 0.5);
	else
		temp = (int)(in + 0.5);

	return temp;
}

double aint (double in)
{
	return((int) in);
}
#endif





/*  ========================================================================  */
/*              bin_search_step_size                                          */
/*  ========================================================================  */
/*
	The following optional code written by Seymour Shlien
	will speed up the outer_loop code which is called
	by iteration_loop. When BIN_SEARCH is defined, the
	outer_loop function precedes the call to the function inner_loop
	with a call to bin_search gain defined below, which
	returns a good starting quantizerStepSize.

	The function count_bits() [a sequence of statements, originally part of inner_loop()]
	was completely rewritten.
*/

int bin_search_StepSize
(
	int						desired_rate,
	double					start,
	int						*abs_ix,
	double					abs_xr[576],
	gr_info					*cod_info
)
{
	double					top, bot, next, last;
	int						bit;

	top = start;
	bot = 200;

	next = start;
	do
	{
		last = next;
		next = aint((top+bot)/2.0);
		cod_info->quantizerStepSize = next;

		tjBitOverflow1 = FALSE;
		quantize_tj (abs_xr, abs_ix, cod_info);
		if (tjBitOverflow1)
			bit = 100000;
		else
			bit = count_bits (abs_ix, cod_info);

		if (bit>desired_rate) 
			top = next;
		else 
			bot = next;
	}
	while ((bit != desired_rate) && (fabs(last-next) > 1.0));

	return next;
}



