#include <vector>
#include <list>
#include <stdexcept>
#include <iterator>
#include <string>
using std::vector;
using std::list;
using std::invalid_argument;
using std::string;
using std::pair;

// Kada klasa Hash musi posiada dwie metody: hash() oraz numBuckets().
template <typename T>
class DefaultHash
{
 public:
  // zgasza wyjtek invalid_argument jeeli numBuckets nie jest dodatnia
  DefaultHash(int numBuckets = 101) throw (invalid_argument);
  int hash(const T& key) const;
  int numBuckets() const { return mNumBuckets; }

 protected:
  int mNumBuckets;
};

// Specjalizacja dla typu string
template <>
class DefaultHash<string>
{
 public:
  // zgasza wyjtek invalid_argument jeeli numBuckets nie jest dodatnia
  DefaultHash(int numBuckets = 101) throw (invalid_argument);
  int hash(const string& key) const;
  int numBuckets() const { return mNumBuckets; }

 protected:
  int mNumBuckets;
};

template <typename Key, typename T, typename Compare, typename Hash>
class hashmap;


// Definicja klasy HashIterator
template<typename Key, typename T, typename Compare, typename Hash>
class HashIterator : public std::iterator<std::bidirectional_iterator_tag,
    pair<const Key, T> >
{
  // Klasa iteratora musi mie dostp do skadnikw protected z hashmap
  friend class hashmap<Key, T, Compare, Hash>;

    public:
        HashIterator(); // Iteratory dwukierunkowe musz mie domylny konstruktor.
        HashIterator(int bucket,
            typename list<pair<const Key, T> >::iterator listIt,
            const hashmap<Key, T, Compare, Hash>* inHashmap);

        pair<const Key, T>& operator*() const;

        // Zwracany typ musi by taki, aby mona byo zastosowa na nim operator -> .
        // Zwraca wskanik do pair<const Key, T>, na ktrym kompikator
        // moe znw zastosowa ->.
        pair<const Key, T>* operator->() const;
        HashIterator<Key, T, Compare, Hash>& operator++();
        const HashIterator<Key, T, Compare, Hash> operator++(int);

        HashIterator<Key, T, Compare, Hash>& operator--();
        const HashIterator<Key, T, Compare, Hash> operator--(int);

        // nie musimy definiowa konstruktora kopiujacego ani operator=, poniewa
        // domylne dziaanie jest prawidowe.

        // nie potrzebujemy destruktora, poniewa domylne dziaanie
        // (nie usuwanie mHashmap) jest prawidowe.

        // Ponisze operatory wystarczaj, poniewa nie obsugujemy porwnywania
        // innych typw.
        bool operator==(const HashIterator& rhs) const;
        bool operator!=(const HashIterator& rhs) const;

    protected:
        int mBucket;
        typename list<pair<const Key, T> >::iterator mIt;
        const hashmap<Key, T, Compare, Hash>* mHashmap;

        // Metody pomocnicze dla operator++ i operator--
        void increment();
        void decrement();
};

template <typename Key, typename T, typename Compare = std::equal_to<Key>,
      typename Hash = DefaultHash<Key> >
class hashmap
{
  public:
  typedef Key key_type;
  typedef T mapped_type;
  typedef pair<const Key, T> value_type;
  typedef Compare key_compare;
  typedef pair<const Key, T>& reference;
  typedef const pair<const Key, T>& const_reference;
  typedef size_t size_type;
  typedef ptrdiff_t difference_type;
  typedef HashIterator<Key, T, Compare, Hash> iterator;
  typedef HashIterator<Key, T, Compare, Hash> const_iterator;

  // Wymagana definicja klasy dla kontenerw asocjacyjnch.
  class value_compare :
  public std::binary_function<value_type, value_type, bool>
  {
    friend class hashmap<Key, T, Compare, Hash>;
  public:
    bool operator() (const value_type& x, const value_type& y) const
      {
    return comp(x.first, y.first);
      }
  protected:
    Compare comp;
    value_compare(Compare c) : comp(c) {}
  };

  // Klasa iteratora musi mie dostp do skadnikw protected z hashmap
  friend class HashIterator<Key, T, Compare, Hash>;

  // Metody iteratora.
  iterator begin();
  iterator end();
  const_iterator begin() const;
  const_iterator end() const;


  // Konstruktor
  // zgasza wyjtek invalid_argument jeeli obiekt hash zawiera niedodatni
  // liczb kubekw.
  explicit hashmap(const Compare& comp = Compare(),
           const Hash& hash = Hash()) throw(invalid_argument);

  template <class InputIterator>
  hashmap(InputIterator first, InputIterator last,
      const Compare& comp = Compare(), const Hash& hash = Hash())
  throw(invalid_argument);

  // destruktor, konstruktor kopiujcy i operator przypisania
  ~hashmap();
  hashmap(const hashmap<Key, T, Compare, Hash>& src);
  hashmap<Key, T, Compare, Hash>& operator=(const hashmap<
                        Key, T, Compare, Hash>& rhs);

  // Metody rozmiaru.
  bool empty() const;
  size_type size() const;
  size_type max_size() const;

  // Metody wstawiania elementw.
  T& operator[] (const key_type& x);
  pair<iterator, bool> insert(const value_type& x);
  iterator insert(iterator position, const value_type& x);
        template <class InputIterator>
  void insert(InputIterator first, InputIterator last);

  // Metody usuwania elementw.
  void erase(iterator position);
  size_type erase(const key_type& x);
  void erase(iterator first, iterator last);


  // Inne metody modyfikujce.
  void swap(hashmap<Key, T, Compare, Hash>& hashIn);

  void clear();

  // Metody dostpu dla zachowania zgodnoci z STL.
  key_compare key_comp() const;
  value_compare value_comp() const;

  // Metody wyszukujce.
  iterator find(const key_type& x);
  const_iterator find(const key_type& x) const;
  size_type count(const key_type& x) const;

protected:
typename list<pair<const Key, T> >::iterator
findElement(const key_type& x, int& bucket) const;

  typedef list<value_type> ListType;

  // W tej pierwszej implementacji atwiej uy wektora
  // zamiast wskanika do wektora, ktry wymaga dynamicznego tworzenia.
  // Jednak uyjemy wskanika do wektora, dziki czemu w ostatecznej
  // implementacji metoda swap() bdzie moga dziaa w staym czasie.
  vector<ListType>* mElems;
  int mSize;
  Compare mComp;
  Hash mHash;
};

#include "hashmap.cpp"
