// zgasza wyjtek invalid_argument jeeli numBuckets nie jest dodatnia
template <typename T>
DefaultHash<T>::DefaultHash(int numBuckets) throw (invalid_argument)
{
  if (numBuckets <= 0) {
    throw (invalid_argument("numBuckets musi by > 0"));
  }
  mNumBuckets = numBuckets;
}

// Korzysta z mieszania przez dzielenie.
// Traktuje klucz jako sekwencj bajtw, sumuje wartoci
// tych bajtw i wylicza reszt z dzielenia tej liczby
// przez liczb kubekw.
template <typename T>
int DefaultHash<T>::hash(const T& key) const
{
  int bytes = sizeof(key);
  unsigned long res = 0;
  for (int i = 0; i < bytes; ++i) {
    res += *((char*)&key + i);
  }
  return (res % mNumBuckets);
}

// zgasza wyjtek invalid_argument jeeli numBuckets nie jest dodatnia
DefaultHash<string>::DefaultHash(int numBuckets) throw (invalid_argument)
{
  if (numBuckets <= 0) {
    throw (invalid_argument("numBuckets musi by > 0"));
  }
  mNumBuckets = numBuckets;
}

// Korzysta z mieszania przez dzielenie po posumowaniu wartoci
// ASCII wszystkich znakw z klucza.
int DefaultHash<string>::hash(const string& key) const
{
  int sum = 0;

  for (size_t i = 0; i < key.size(); i++) {
    sum += key[i];
  }
  return (sum % mNumBuckets);
}

// Tworzy mElems z podan liczb kubekw.
template <typename Key, typename T, typename Compare, typename Hash>
hashmap<Key, T, Compare, Hash>::hashmap(
                    const Compare& comp, const Hash& hash) throw(invalid_argument) :
  mSize(0), mComp(comp), mHash(hash)
{
  if (mHash.numBuckets() <= 0) {
    throw (invalid_argument("Liczba kubekw musi by wiksza od 0"));
  }
  mElems = new vector<list<value_type> >(mHash.numBuckets());
}

template <typename Key, typename T, typename Compare, typename Hash>
hashmap<Key, T, Compare, Hash>::~hashmap()
{
  delete mElems;
}

template <typename Key, typename T, typename Compare, typename Hash>
hashmap<Key, T, Compare, Hash>::hashmap(const hashmap<
                    Key, T, Compare, Hash>& src) :
  mSize(src.mSize), mComp(src.mComp), mHash(src.mHash)
{
  // Nie trzeba sprawdza, czy numBuckets ma warto dodatni
  // poniewa ju to wczeniej zrobilimy w src.

  // Uycie konstruktora kopiujcego klasy vector
  mElems = new vector<list<value_type> >(*(src.mElems));
}

template <typename Key, typename T, typename Compare, typename Hash>
hashmap<Key, T, Compare, Hash>& hashmap<Key, T, Compare, Hash>::operator=(
                                      const hashmap<Key, T, Compare, Hash>& rhs)
{
  // zabezpieczenie przed przypisaniem do samego siebie
  if (this != &rhs) {
    delete mElems;
    mSize = rhs.mSize;
    mComp = rhs.mComp;
    mHash = rhs.mHash;
    // Nie trzeba sprawdza, czy numBuckets ma warto dodatni
    // poniewa ju to wczeniej zrobilimy w rhs.

    // Uycie konstruktora kopiujcego klasy vector
    mElems = new vector<list<value_type> >(*(rhs.mElems));
  }
}

template <typename Key, typename T, typename Compare, typename Hash>
typename list<pair<const Key, T> >::iterator
hashmap<Key, T, Compare, Hash>::findElement(const key_type& x, int& bucket) const
{
  // do uzyskania kubeka obliczamy warto mieszajc
  bucket = mHash.hash(x);

  // Wyszukiwanie klucza w kubekach
  for (typename ListType::iterator it = (*mElems)[bucket].begin();
       it != (*mElems)[bucket].end(); ++it) {
    if (mComp(it->first, x)) {
      return (it);
    }
  }
  return ((*mElems)[bucket].end());
}

template <typename Key, typename T, typename Compare, typename Hash>
typename hashmap<Key, T, Compare, Hash>::value_type*
hashmap<Key, T, Compare, Hash>::find(const key_type& x)
{
  int bucket;
  // Uycie pomocniczej funkcji findElement()
  typename ListType::iterator it = findElement(x, bucket);
  if (it == (*mElems)[bucket].end()) {
    // nie znaleziono elementu -- zwracamy NULL
    return (NULL);
  }
  // Znaleziono element. Zwracamy wskanik do niego.
  return (&(*it));
}

template <typename Key, typename T, typename Compare, typename Hash>
T& hashmap<Key, T, Compare, Hash>::operator[] (const key_type& x)
{
  // Prba znalezienia elementu.
  // Jeeli nie istnieje, dodanie nowego elementu.
  value_type* found = find(x);
  if (found == NULL) {
    insert(make_pair(x, T()));
    found = find(x);
  }
  return (found->second);
}

template <typename Key, typename T, typename Compare, typename Hash>
void hashmap<Key, T, Compare, Hash>::insert(const value_type& x)
{
  int bucket;
  // Prba znalezienia elementu.
  typename ListType::iterator it = findElement(x.first, bucket);

  if (it != (*mElems)[bucket].end()) {
    // Element istnieje.
    return;
  } else {
    // Nie znaleziono elementu, wic jest wstawiany nowy.
    mSize++;
    (*mElems)[bucket].insert((*mElems)[bucket].end(), x);
  }
}

template <typename Key, typename T, typename Compare, typename Hash>
void
hashmap<Key, T, Compare, Hash>::erase(const key_type& x)
{
  int bucket;

  // Prba znalezienia elementu.
  typename ListType::iterator it = findElement(x, bucket);

  if (it != (*mElems)[bucket].end()) {
    // Element istnieje -- kasowanie.
    (*mElems)[bucket].erase(it);
    mSize--;
  }
}
