#include <stdexcept>
#include <vector>
using std::vector;

//
// Szablon klasy RoundRobin
//
// Realizuje prosty bufor cykliczny na licie elementw.
// Klienci dodaj elementy na koniec listy za pomoc metody add().
//
// getNext() zwraca nastpny element z listy, zaczynajc od pierwszego
// i wracajc do pierwszego, gdy zostanie osignity koniec listy.
//
// remove() usuwa element rwny argumentowi.
//
template <typename T>
class RoundRobin
{
 public:
  //
  // Dla zwikszenia wydajnoci klienci mog okrela liczb 
  // spodziewanych elementw.
  //
  RoundRobin(int numExpected = 0);
  ~RoundRobin();

  //
  // Docza element na koniec listy. Moe by wywoywana pomidzy
  // wywoaniami getNext().
  //
  void add(const T& elem);

  //
  // Usuwa pierwszy (i tylko pierwszy) element
  // z listy, ktry jest rwny (w sensie operator==) podanemu elementowi.
  // Moe by wywoywana pomidzy wywoaniami getNext().
  //
  void remove(const T& elem);

  //
  // Zwraca nastpny element z listy, zaczynajc od 0 i wykonujc
  // kolejne cykle, biorc pod uwag dodawane lub usuwane elementy.
  //
  T& getNext() throw(std::out_of_range);
 protected:
  vector<T> mElems;
  typename std::vector<T>::iterator mCurElem;

 private:
  // zapobiega przypisywaniu i przekazwaniu przez referencj
  RoundRobin(const RoundRobin& src);
  RoundRobin& operator=(const RoundRobin& rhs);
};

template <typename T>
RoundRobin<T>::RoundRobin(int numExpected)
{
  // Jeeli klient przekae wskazwk, rezerwujemy tyle miejsca.
  mElems.reserve(numExpected);

  // Inicjalizacja mCurElem pomimo tego, e nie jest wykorzystywana
  // dopki nie znajdzie si co najmniej jeden element.
  mCurElem = mElems.begin();
}

template <typename T>
RoundRobin<T>::~RoundRobin()
{
  // nic nie robimy -- wektor skasuje wszystkie elementy.
}

//
// Zawsze dodajemy nowy element na koniec.
//
template <typename T>
void RoundRobin<T>::add(const T& elem)
{
  //
  // Pomimo tego, e dodajemy elementy na koniec,
  // wektor moe realokowa pami i uniewani iterator.
  // Skorzystamy z cech iteratora o swobodnym dostpie 
  // do zapamitania miejsca.
  //
  int pos = mCurElem - mElems.begin();

  // dodanie elementu
  mElems.push_back(elem);

  // Jeeli jest to pierwszy element, inicjujemy iterator pocztkiem.
  if (mElems.size() == 1) { 
    mCurElem = mElems.begin();
  } else {
    // ustawiamy na zapamitane miejsce
    mCurElem = mElems.begin() + pos;
  }
}

template <typename T>
void RoundRobin<T>::remove(const T& elem)
{
  for (typename std::vector<T>::iterator it = mElems.begin(); it != mElems.end(); ++it) {
    if (*it == elem) {
      //
      // Usuwanie elementu uniewania nasz iterator mCurElem, jeeli 
      // odwouje si do elementu poza punktem usuwania.
      // Skorzystamy z cech iteratora o swobodnym dostpie 
      // do ledzenia pozycji biecego elementu po usuwaniu.
      //
      int newPos;

      // Jeeli biecy iterator znajduje si przed lub na usuwanym elemencie
      // nowa pozycja jest taka sama jak poprzednio.
      if (mCurElem <= it) {
    newPos = mCurElem - mElems.begin();
      } else {
    // w przeciwmym razie jest o jeden mniejsza ni poprzednio
    newPos = mCurElem - mElems.begin() - 1;
      }
      // usuwamy element (i ignorujemy zwracan warto)
      mElems.erase(it);

      // Ponownie ustawiamy iterator
      mCurElem = mElems.begin() + newPos;

      // Jeeli wskazuje na ostatni element i zostanie on usunity
      // musimy wrci do pierwszego.
      if (mCurElem == mElems.end()) {
    mCurElem = mElems.begin();
      }
      return;
    }
  }
}

template <typename T>
T& RoundRobin<T>::getNext()throw(std::out_of_range)
{
  // Na pocztku upewniamy si, e istniej jakiekolwiek elementy.
  if (mElems.empty()) {
    throw std::out_of_range("Brak elementw w licie");
  }

  // pobranie referencji do zwrcenia
  T& retVal = *mCurElem;

  // Zwikszenie iteratora modulo liczba elementw
  ++mCurElem;
  if (mCurElem == mElems.end()) {
    mCurElem = mElems.begin();
  }

  // zwrcenie referencji
  return (retVal);
}
