#include "NameDB.h"

#include <fstream>

using namespace std;

//
// Odczytuje nazwy z pliku i wypenia baz danych.
// Baza danych jest wektorem par nazwa-liczba przechowujcym
// liczb wystpie poszczeglnych imion w pliku tekstowym.
//
NameDB::NameDB(const string& nameFile) throw (invalid_argument)
{
  // Otwrz plik i sprawd, czy nie pojawi si bd.
  ifstream inFile(nameFile.c_str());
  if (!inFile) {
    throw invalid_argument("Nieudane otwarcie pliku.\n");
  }

  // Odczytywanie kolejnych imion.
  string name;
  while (inFile >> name) {
    // Sprawd, czy imi znajduje si ju w baze danych.
    if (nameExists(name)) {
      // Jeli znajduje si w bazie danych, po prostu 
      // zwiksz licznik.
      incrementNameCount(name);
    } else {
      // Jeli to nowe imi, dodaj je do bazy z 
      // licznikiem wynoszcym 1.
      addNewName(name);
    }
  }
  inFile.close();
}

//
// nameExists
//
// Zwrca true, jeli imi istnieje w bazie danych. W przeciwnym przypadku zwraca false.
//
bool NameDB::nameExists(const string& name) const
{
  // Dokonaj przejcia przez wektor, poszukujc imienia.
  for (vector<pair<string, int> >::const_iterator it = mNames.begin();
       it !=  mNames.end(); ++it) {
    if (it->first == name) {
      return (true);
    }
  }
  return (false);
}

//
// incrementNameCount
//
// Warunki wstpne: imi istnieje w wektorze imion.
// Warunki kocowe: inkrementacja licznika zwizanego z danym imieniem.
//
void NameDB::incrementNameCount(const string& name)
{
  for (vector<pair<string, int> >::iterator it = mNames.begin();
       it != mNames.end(); ++it) {
    if (it->first == name) {
      it->second++;
      return;
    }
  }
}

//
// addNewName
//
// Dodaje nowe imi do bazy danych.
//
void NameDB::addNewName(const string& name)
{
  mNames.push_back(make_pair<string, int>(name, 1));
}

//
// getNameRank
//
// Zwraca ranking imienia.
// Najpierw wyszukuje imi i pobiera liczb dzieci i takim imieniu.
// Nastpnie przechodzi przez wszystkie imiona i zlicza te, ktre maj liczb
// wiksz od szukanego imienia. Zwraca zliczon warto jako ranking.
//
int NameDB::getNameRank(const string& name) const
{
  // Wykorzystuje metod getAbsoluteNumber().
  int num = getAbsoluteNumber(name);

  // Sprawdzenie, czy imi zostao odnalzione.
  if (num == -1) {
    return (-1);
  }

  //
  // Zliczenie wszystkich imion w wektorze, ktre maj wysz warto
  // licznika ni sprawdzane imi. Jeli adne imi nie ma wikszej wartoci,
  // wtedy szukane imi na ranking 1. Kade znalezione imi z wysz wartoci
  // licznika zmniejsza pozycj w rankingu o jeden.
  //
  int rank = 1;
  for (vector<pair<string, int> >::const_iterator it = mNames.begin();
       it != mNames.end(); ++it) {
    if (it->second > num) {
      rank++;
    }
  }

  return (rank);
}

//
// getAbsoluteNumber
//
// Zwraca licznik powizany z dan nazw.
//
int NameDB::getAbsoluteNumber(const string& name) const
{
  for (vector<pair<string, int> >::const_iterator it = mNames.begin();
       it != mNames.end(); ++it) {
    if (it->first == name) {
      return(it->second);
    }
  }
  return (-1);
}
