#include <iterator>

// zgasza wyjtek invalid_argument jeeli numBuckets nie jest dodatnia
template <typename T>
DefaultHash<T>::DefaultHash(int numBuckets) throw (invalid_argument)
{
  if (numBuckets <= 0) {
    throw (invalid_argument("Liczba kubekw musi by wiksza od 0"));
  }
  mNumBuckets = numBuckets;
}

// Korzysta z mieszania przez dzielenie.
// Traktuje klucz jako sekwencj bajtw, sumuje wartoci
// tych bajtw i wylicza reszt z dzielenia tej liczby
// przez liczb kubekw.
template <typename T>
int DefaultHash<T>::hash(const T& key) const
{
  int bytes = sizeof(key);
  unsigned long res = 0;
  for (int i = 0; i < bytes; ++i) {
    res += *((char*)&key + i);
  }
  return (res % mNumBuckets);
}

// zgasza wyjtek invalid_argument jeeli numBuckets nie jest dodatnia
DefaultHash<string>::DefaultHash(int numBuckets) throw (invalid_argument)
{
  if (numBuckets <= 0) {
    throw (invalid_argument("Liczba kubekw musi by wiksza od 0"));
  }
  mNumBuckets = numBuckets;
}

// Korzysta z mieszania przez dzielenie po posumowaniu wartoci
// ASCII wszystkich znakw z klucza.
int DefaultHash<string>::hash(const string& key) const
{
  int sum = 0;

  for (size_t i = 0; i < key.size(); i++) {
    sum += key[i];
  }
  return (sum % mNumBuckets);
}

// Dereferencja lub inkrementacja iteratora utworzonego za pomoc
// domylnego konstruktora jest niezdefiniowana, wic nie ma
// znaczenia jak zwrcimy warto.
template<typename Key, typename T, typename Compare, typename Hash>
HashIterator<Key, T, Compare, Hash>::HashIterator()
{
  mBucket = -1;
  mIt = list<pair<const Key, T> >::iterator();
  mHashmap = NULL;
}

template<typename Key, typename T, typename Compare, typename Hash>
HashIterator<Key, T, Compare, Hash>::HashIterator(int bucket,
    typename list<pair<const Key, T> >::iterator listIt,
    const hashmap<Key, T, Compare, Hash>* inHashmap) :
  mBucket(bucket), mIt(listIt), mHashmap(inHashmap)
{
}

// Zwraca biecy element.
template<typename Key, typename T, typename Compare, typename Hash>
pair<const Key, T>& HashIterator<Key, T, Compare, Hash>::operator*() const
{
  return (*mIt);
}

// Zwraca iterator, wic kompilator moe wykona operacj -> aby odwoa si
// do odpowiedniego pola.
template<typename Key, typename T, typename Compare, typename Hash>
pair<const Key, T>*
HashIterator<Key, T, Compare, Hash>::operator->() const
{
  return (&(*mIt));
}
// Faktyczne dziaanie jest wykonywane w metodzie pomocniczej increment()
template<typename Key, typename T, typename Compare, typename Hash>
HashIterator<Key, T, Compare, Hash>&
HashIterator<Key, T, Compare, Hash>::operator++()
{
  increment();
  return (*this);
}

// Faktyczne dziaanie jest wykonywane w metodzie pomocniczej increment()
template<typename Key, typename T, typename Compare, typename Hash>
const HashIterator<Key, T, Compare, Hash>
HashIterator<Key, T, Compare, Hash>::operator++(int)
{
  HashIterator<Key, T, Compare, Hash> oldIt = *this;
  increment();
  return (oldIt);
}

// Faktyczne dziaanie jest wykonywane w metodzie pomocniczej decrement()
template<typename Key, typename T, typename Compare, typename Hash>
HashIterator<Key, T, Compare, Hash>&
HashIterator<Key, T, Compare, Hash>::operator--()
{
  decrement();
  return (*this);
}

// Faktyczne dziaanie jest wykonywane w metodzie pomocniczej decrement()
template<typename Key, typename T, typename Compare, typename Hash>
const HashIterator<Key, T, Compare, Hash>
HashIterator<Key, T, Compare, Hash>::operator--(int)
{
  HashIterator<Key, T, Compare, Hash> newIt = *this;
  decrement();
  return (newIt);
}

// Dziaanie jest niezdefiniowane jeeli mIt odwouje si poza kocowy
// element tabeli lub jest w inny sposb niewaciwa.
template<typename Key, typename T, typename Compare, typename Hash>
void HashIterator<Key, T, Compare, Hash>::increment()
{
  // mIt jest iteratorem do pojedynczego kubeka.
  // Zwikszamy warto.
  ++mIt;

  // Jeeli znajduje si na kocu biecego kubeka,
  // znajd nastpny kubeek z elementami.
  if (mIt == (*mHashmap->mElems)[mBucket].end()) {
    for (size_t i = mBucket + 1; i < (*mHashmap->mElems).size(); i++) {
      if (!((*mHashmap->mElems)[i].empty())) {
    // Znaleziono niepusty element.
    // mIt bdzie odwoywaa si do jego pierwszego elementu.
    mIt = (*mHashmap->mElems)[i].begin();
    mBucket = i;
    return;
      }
    }
    // Brak pustych elementw. Przypisz mIt kocowy
    // iterator z ostatniej listy.
    mBucket = (*mHashmap->mElems).size() - 1;
    mIt = (*mHashmap->mElems)[mBucket].end();
  }
}
// Dziaanie jest niezdefiniowane jeeli mIt odwouje si do pierwszego
// elementu tabeli lub jest w inny sposb niewaciwa.
template<typename Key, typename T, typename Compare, typename Hash>
void HashIterator<Key, T, Compare, Hash>::decrement()
{
  // mIt jest iteratorem do pojedynczego kubeka.
  // Jeeli znajduje si na pocztku biecego kubeka,
  // nie zmniejszaj zmiennej. Zamiast tego sprbuj znale niepusty
  // kubeek przed nim.
  if (mIt == (*mHashmap->mElems)[mBucket].begin()) {
    for (int i = mBucket - 1; i >= 0; --i) {
      if (!((*mHashmap->mElems)[i].empty())) {
    mIt = (*mHashmap->mElems)[i].end();
    --mIt;
    mBucket = i;
    return;
      }
    }
    // Brak niepistych kubekw. Jest to niewaciwa dekrementacja.
    // Przypisz mIt do elementu o jeden wczeniejszego ni pocztkowy
    // element pierwszej listy (nieprawidowa pozycja).
    mIt = (*mHashmap->mElems)[0].begin();
    --mIt;
    mBucket = 0;
  } else {
    // Nie jestemy na pocztku kubeka, wic przechodzimy w d.
    --mIt;
  }
}

template<typename Key, typename T, typename Compare, typename Hash>
bool HashIterator<Key, T, Compare, Hash>::operator==(
                             const HashIterator& rhs) const
{
  // Wszystkie pola, w tym hashmap do ktrego odwouj si iteratory
  // musz by sobie rwne.
  return (mHashmap == rhs.mHashmap && mBucket == rhs.mBucket &&
      mIt == rhs.mIt);
}

template<typename Key, typename T, typename Compare, typename Hash>
bool HashIterator<Key, T, Compare, Hash>::operator!=(
                             const HashIterator& rhs) const
{
  return (!operator==(rhs));
}


// Tworzy mElems z podan liczb kubekw.
template <typename Key, typename T, typename Compare, typename Hash>
hashmap<Key, T, Compare, Hash>::hashmap(
                    const Compare& comp, const Hash& hash) throw(invalid_argument) :
  mSize(0), mComp(comp), mHash(hash)
{
  if (mHash.numBuckets() <= 0) {
    throw (invalid_argument("Liczba kubekw musi by wiksza od 0"));
  }
  mElems = new vector<list<value_type> >(mHash.numBuckets());
}

// Wstawienie elementw odbywa si faktycznie w insert().
template <typename Key, typename T, typename Compare, typename Hash>
template <class InputIterator>
hashmap<Key, T, Compare, Hash>::hashmap(
InputIterator first, InputIterator last, const Compare& comp,
const Hash& hash) throw(invalid_argument) : mSize(0), mComp(comp), mHash(hash)
{
  if (mHash.numBuckets() <= 0) {
    throw (invalid_argument("Liczba kubekw musi by wiksza od 0"));
  }
  mElems = new vector<list<value_type> >(mHash.numBuckets());
  insert(first, last);
}


template <typename Key, typename T, typename Compare, typename Hash>
hashmap<Key, T, Compare, Hash>::~hashmap()
{
  delete mElems;
}

template <typename Key, typename T, typename Compare, typename Hash>
hashmap<Key, T, Compare, Hash>::hashmap(const hashmap<
                    Key, T, Compare, Hash>& src) :
  mSize(src.mSize), mComp(src.mComp), mHash(src.mHash)
{
  // Nie trzeba sprawdza, czy numBuckets ma warto dodatni
  // poniewa ju to wczeniej zrobilimy w src.

  // Uycie konstruktora kopiujcego klasy vector
  mElems = new vector<list<value_type> >(*(src.mElems));
}

template <typename Key, typename T, typename Compare, typename Hash>
hashmap<Key, T, Compare, Hash>& hashmap<Key, T, Compare, Hash>::operator=(
                                      const hashmap<Key, T, Compare, Hash>& rhs)
{
  // zabezpieczenie przed przypisaniem do samego siebie
  if (this != &rhs) {
    delete mElems;
    mSize = rhs.mSize;
    mComp = rhs.mComp;
    mHash = rhs.mHash;
    // Nie trzeba sprawdza, czy numBuckets ma warto dodatni
    // poniewa ju to wczeniej zrobilimy w rhs.

    // Uycie konstruktora kopiujcego klasy vector
    mElems = new vector<list<value_type> >(*(rhs.mElems));
  }
  return (*this);
}

template <typename Key, typename T, typename Compare, typename Hash>
typename list<pair<const Key, T> >::iterator
hashmap<Key, T, Compare, Hash>::findElement(const key_type& x, int& bucket) const
{
  // do uzyskania kubeka obliczamy warto mieszajc
  bucket = mHash.hash(x);

  // Wyszukiwanie klucza w kubekach
  for (typename ListType::iterator it = (*mElems)[bucket].begin();
       it != (*mElems)[bucket].end(); ++it) {
    if (mComp(it->first, x)) {
      return (it);
    }
  }
  return ((*mElems)[bucket].end());
}

template <typename Key, typename T, typename Compare, typename Hash>
typename hashmap<Key, T, Compare, Hash>::iterator
hashmap<Key, T, Compare, Hash>::find(const key_type& x)
{
  int bucket;
  // Uycie pomocniczej funkcji findElement()
  typename ListType::iterator it = findElement(x, bucket);
  if (it == (*mElems)[bucket].end()) {
    // nie znaleziono elementu -- zwracamy iterator kocowy
    return (end());
  }
  // Znaleziono element -- konwertujemy kubeek/iterator na HashIterator.
  return (HashIterator<Key, T, Compare, Hash>(bucket, it, this));
}

template <typename Key, typename T, typename Compare, typename Hash>
typename hashmap<Key, T, Compare, Hash>::const_iterator
hashmap<Key, T, Compare, Hash>::find(const key_type& x) const
{
  int bucket;
  // Uycie pomocniczej funkcji findElement()
  typename ListType::iterator it = findElement(x, bucket);
  if (it == (*mElems)[bucket].end()) {
    // nie znaleziono elementu -- zwracamy iterator kocowy
    return (end());
  }
  // Znaleziono element -- konwertujemy kubeek/iterator na HashIterator.
  return (HashIterator<Key, T, Compare, Hash>(bucket, it, this));
}

template <typename Key, typename T, typename Compare, typename Hash>
typename hashmap<Key, T, Compare, Hash>::size_type
hashmap<Key, T, Compare, Hash>::count(const key_type& x) const
{
  // Wystpuje to 1 lub 0 elementw pasujacych do klucza x.
  // Jeeli znajdziemy klucz, zwracamy 1, w przeciwnym razie 0.
  if (find(x) == end()) {
    return (0);
  } else {
    return (1);
  }
}
template <typename Key, typename T, typename Compare, typename Hash>
T& hashmap<Key, T, Compare, Hash>::operator[] (const key_type& x)
{
  // Ta definicja jest taka sama jak uyta w map,
  // zgodnie ze standardem.
  // Jest nieco zoona -- wykonuje wstawienie
  // nowej pary klucz-warto z x i nowa wartoci. Niezalenie od tego,
  // czy wstawienie si udaoo, insert() zwraca par iterator-bool.
  // Iterator odwouje si do pary klucz-warto.
  // Element second jest wartoci do zwrcenia.
  return (((insert(make_pair(x, T()))).first)->second);
}

template <typename Key, typename T, typename Compare, typename Hash>
pair<typename hashmap<Key, T, Compare, Hash>::iterator, bool>
hashmap<Key, T, Compare, Hash>::insert(const value_type& x)
{
  int bucket;
  // Prba znalezienia elementu.
  typename ListType::iterator it = findElement(x.first, bucket);

  if (it != (*mElems)[bucket].end()) {
    // Element istnieje.
    // Konwersja iteratora listy na HashIterator, ktry
    // wymaga rwnie kubeka i wskanika do hashmap.
    HashIterator<Key, T, Compare, Hash> newIt(bucket, it, this);

    // Niektre kompilatory nie lubia tutaj make_pair.
    pair<HashIterator<Key, T, Compare, Hash>, bool> p(newIt, false);
    return (p);
  } else {
    // Nie znalelimy elementu, wic wstawiamy nowy.
    mSize++;
    typename ListType::iterator endIt =
      (*mElems)[bucket].insert((*mElems)[bucket].end(), x);
    pair<HashIterator<Key, T, Compare, Hash>, bool> p(
      HashIterator<Key, T, Compare, Hash>(bucket, endIt, this), true);
    return (p);
  }
}

template <typename Key, typename T, typename Compare, typename Hash>
typename hashmap<Key, T, Compare, Hash>::iterator
    hashmap<Key, T, Compare, Hash>::insert(typename hashmap<Key, T, Compare,
    Hash>::iterator position, const value_type& x)
{
  // cakowicie ignorujemy parametr position
  return (insert(x).first);
}

template <typename Key, typename T, typename Compare, typename Hash>
template <class InputIterator>
void hashmap<Key, T, Compare, Hash>::insert(InputIterator first,
                        InputIterator last)
{
  // Kopiowanie wszystkich elementw z zakresu przez uycie adaptera insert_iterator.
  // Przekazujemy begin() jako pozycj -- insert() i tak j zignoruje.
  // anyway.
  std::insert_iterator<hashmap<Key, T, Compare, Hash> > it(*this, begin());
  copy(first, last, it);
}

template <typename Key, typename T, typename Compare, typename Hash>
typename hashmap<Key, T, Compare, Hash>::size_type
hashmap<Key, T, Compare, Hash>::erase(const key_type& x)
{
  int bucket;

  // Prba znalezienia elementu.
  typename ListType::iterator it = findElement(x, bucket);

  if (it != (*mElems)[bucket].end()) {
    // Element istnieje -- kasowanie.
    (*mElems)[bucket].erase(it);
    mSize--;
    return (1);
  } else {
    return (0);
  }
}

template <typename Key, typename T, typename Compare, typename Hash>
void hashmap<Key, T, Compare, Hash>::erase(
                       hashmap<Key, T, Compare, Hash>::iterator position)
{
  // Kasowanie elementu z kubeka.
  (*mElems)[position.mBucket].erase(position.mIt);
  mSize--;
} 

template <typename Key, typename T, typename Compare, typename Hash>
void hashmap<Key, T, Compare, Hash>::erase(
                       hashmap<Key, T, Compare, Hash>::iterator first,
                       hashmap<Key, T, Compare, Hash>::iterator last)
{
  typename hashmap<Key, T, Compare, Hash>::iterator cur, next;

  // Kasowanie wszystkich elementw z zakresu.
  for (next = first; next != last; ) {
    cur = next++;
    erase(cur);
  }
}

template <typename Key, typename T, typename Compare, typename Hash>
void hashmap<Key, T, Compare, Hash>::clear()
{
  // Wywoanie clear na kadej licie
  for_each(mElems->begin(), mElems->end(), mem_fun_ref(&ListType::clear));
  mSize = 0;
}


template <typename Key, typename T, typename Compare, typename Hash>
bool hashmap<Key, T, Compare, Hash>::empty() const
{
  return (mSize == 0);
}

template <typename Key, typename T, typename Compare, typename Hash>
typename hashmap<Key, T, Compare, Hash>::size_type
hashmap<Key, T, Compare, Hash>::size() const
{
  return (mSize);
}

template <typename Key, typename T, typename Compare, typename Hash>
typename hashmap<Key, T, Compare, Hash>::size_type
hashmap<Key, T, Compare, Hash>::max_size() const
{
  // W najgorszym przypadku wszystkie elementy trafi do tej samej listy
  // wic max_size jest rwne max_size jednej listy.
  // W kodzie zakadamy, e wszystkie listy maj tak sam warto
  // max_size.
  return ((*mElems)[0].max_size());
}

// Po prostu zamieniami cztery zmienne skadowe
// Korzystamy z oglnego szablonu swap.
template <typename Key, typename T, typename Compare, typename Hash>
void hashmap<Key, T, Compare, Hash>::swap(hashmap<
                      Key, T, Compare, Hash>& hashIn)
{
  // Jawna kwalifikacja za pomoc std:: aby kompilator nie uzna
  // e jest to wywoanie rekurencyjne.
  std::swap(*this, hashIn);
}

template <typename Key, typename T, typename Compare, typename Hash>
typename hashmap<Key, T, Compare, Hash>::iterator
hashmap<Key, T, Compare, Hash>::begin()
{
  if (mSize == 0) {
    // Specjalny przypadek: brak elementw, wic zwracamy iterator kocowy.
    return (end());
  }

  // Wiemy, e jest co najmniej jeden element. Wyszukanie pierwszego elementu.
  for (size_t i = 0; i < mElems->size(); ++i) {
    if (!((*mElems)[i].empty())) {
      return (HashIterator<Key, T, Compare, Hash>(i,
                          (*mElems)[i].begin(), this));
    }
  }
  // Kod nie powinien tu dotrze, ale jeeli tak si zdarzy, zwracamy iterator kocowy.
  return (end());
}

template <typename Key, typename T, typename Compare, typename Hash>
typename hashmap<Key, T, Compare, Hash>::iterator
hashmap<Key, T, Compare, Hash>::end()
{
  // Iterator kocowy jest iteratorem kocowym z listy ostatniego kubeka.
  return (HashIterator<Key, T, Compare, Hash>(mElems->size() - 1,
    (*mElems)[mElems->size() - 1].end(), this));
}

template <typename Key, typename T, typename Compare, typename Hash>
typename hashmap<Key, T, Compare, Hash>::iterator
hashmap<Key, T, Compare, Hash>::begin() const
{
  if (mSize == 0) {
    // Specjalny przypadek: brak elementw, wic zwracamy iterator kocowy.
    return (end());
  }

  // Wiemy, e jest co najmniej jeden element. Wyszukanie pierwszego elementu.
  for (size_t i = 0; i < mElems->size(); ++i) {
    if (!((*mElems)[i].empty())) {
      return (HashIterator<Key, T, Compare, Hash>(i,
                          (*mElems)[i].begin(), this));
    }
  }
  // Kod nie powinien tu dotrze, ale jeeli tak si zdarzy, zwracamy iterator kocowy.
  return (end());
}

template <typename Key, typename T, typename Compare, typename Hash>
typename hashmap<Key, T, Compare, Hash>::iterator
hashmap<Key, T, Compare, Hash>::end() const
{
  // Iterator kocowy jest iteratorem kocowym z listy ostatniego kubeka.
  return (HashIterator<Key, T, Compare, Hash>(mElems->size() - 1,
    (*mElems)[mElems->size() - 1].end(), this));
}
