#include <algorithm>
#include <functional>
#include <map>
#include <list>
#include <iostream>
#include <utility>
#include <string>
using namespace std;

void printString(const string& str)
{
  cout << " {" << str << "}";
}

void printCounty(const pair<const string, list<string> >& county)
{
  cout << county.first << ":";
  for_each(county.second.begin(), county.second.end(), &printString);
  cout << endl;
}

//
// NameInList
//
// Functor sprawdzajcy, czy cig znajduje si w licie cigw 
// (przekazywanych w czasie tworzenia).
//
class NameInList : public unary_function<string, bool>
{
public:
  NameInList(const list<string>& names) : mNames(names) {}
  bool operator() (const string& val);

protected:
  const list<string>& mNames;
};

//
// operatory wywoania funkcji dla funktora NameInList
//
// Zwraca true jeeli warto name zostanie znaleziona w mNames, 
// w przeciwnym razie false. Korzysta z algorytmu find().
//
bool NameInList::operator() (const string& name)
{
  return (find(mNames.begin(), mNames.end(), name) != mNames.end());
}


//
// RemoveNames
//
// Klasa funktora, ktra korzysta z par string/list<string> i usuwa
// z list wszystkie cigi znajdujce si w licie names
// (ustawianej w konstruktorze).
//
class RemoveNames : public unary_function<pair<const string, list<string> >,
    void>
{
public:
  RemoveNames(const list<string>& names) : mNames(names) {}
  void operator() (pair<const string, list<string> >& val);

protected:
  const list<string>& mNames; 
};



//
// Operator wywoania funkcji dla funktora RemoveNames.
//
// Korzysta z remove_if() oraz erase w celu trwaego usunicia nazwisk
// z listy.
//
// Nazwiska s usuwane jeeli znajduj si w licie mNames. Do sprawdzenia, 
// czy nazwisko znajduje si w licie naley uy funktora NameInList.
//
void RemoveNames::operator() (pair<const string, list<string> >& val)
{
  list<string>::iterator it = remove_if(val.second.begin(), val.second.end(),
                    NameInList(mNames));
  val.second.erase(it, val.second.end());
} 

//
// getDuplicates()
//
// Zwraca list nazwisk, ktre wystpiy w wicej niz jednej licie 
// znajdujcej si w kontenerze map.
//
// W tym rozwizaniu generowana jest jedna, dua lista nazwisk z 
// wszystkich list z map, jest ona sortowana, a nastpnie 
// za pomoc adjacent_find() wyszukiwane s powtrzenia.
//
list<string> getDuplicates(const map<string, list<string> >& voters)
{
  list<string> allNames, duplicates;

  // Umieszczenie nazwisk z wszystkich list w jednej, duej licie.
  map<string, list<string> >::const_iterator it;
  for(it = voters.begin(); it != voters.end(); ++it) {
    allNames.insert(allNames.end(), it->second.begin(), it->second.end());
  }

  // Sortowanie listy -- uyta zostaa specjalizacja sort z kontenera list,
  // poniewa jest szybsza od oglnego algorytmu.
  allNames.sort();

  //
  // Po posortowaniu powtrzone nazwiska bd obok siebie.
  // Korzystamy z adjacent_find() do znalezienia dwch lub wicej identycznych
  // ssiadujcych ze sob nazwisk.
  //
  // Loop until adjacent_find returns the end iterator.
  //
  list<string>::iterator lit;
  for (lit = allNames.begin(); lit != allNames.end(); ++lit) {
    lit = adjacent_find(lit, allNames.end());
    if (lit == allNames.end()) {
      break;
    }
    duplicates.push_back(*lit);
  }

  //
  // Jeeli kto by na wicej ni jednej licie wyborcw, 
  // znajdzie si wicej ni raz na licie duplikatw. List t sortujemy
  // i usuwamy duplikaty za pomoc unique.
  //
  // Korzystamy ze specjalizacji dla kontenera list, poniewa s szybsze ni
  // oglne algorytmy.
  //
  duplicates.sort();
  duplicates.unique();

  return (duplicates);
}


//
// auditVoterRolls
//
// Oczekuje par string/list<string> pairs w ktrych kluczem jest nazwa okrgu
// ktre zawieraj listy wszystkich zarejestrowanych wyborcw.
//
// Usuwa z listy wszystkie nazwiska znajdujce si w convictedFelons list 
// oraz wszystkie nazwiska znajdujce si na dowolnej innej licie.
//
void auditVoterRolls(map<string, list<string> >& votersByCounty,
             const list<string>& convictedFelons)
{
  // pobranie wszystkich powielonych nazwisk
  list<string> duplicates = getDuplicates(votersByCounty);

  // poczenie duplikatw i skazacw -- chcemy 
  // usun z rejestrw wyborcw nazwiska z obu list
  duplicates.insert(duplicates.end(), convictedFelons.begin(),
            convictedFelons.end());

  // Jeeli znajduj si duplikaty, usuwamy je.
  // Uyj sort i uniquw dla listy a nie oglnych algorytmw
  // poniewa metody list s bardziej wydajne.
  duplicates.sort();
  duplicates.unique();

  // Teraz usu wszystkie nazwy do skasowania.
  for_each(votersByCounty.begin(), votersByCounty.end(),
       RemoveNames(duplicates));
}

int main(int argc, char** argv)
{
  map<string, list<string> > voters;
  list<string> nameList, felons;
  nameList.push_back("Amy Aardvark");
  nameList.push_back("Bob Buffalo");
  nameList.push_back("Charles Cat");
  nameList.push_back("Dwayne Dog");

  voters.insert(make_pair("Orange", nameList));

  nameList.clear();
  nameList.push_back("Elizabeth Elephant");
  nameList.push_back("Fred Flamingo");
  nameList.push_back("Amy Aardvark");

  voters.insert(make_pair("Los Angeles", nameList));

  nameList.clear();
  nameList.push_back("George Goose");
  nameList.push_back("Heidi Hen");
  nameList.push_back("Fred Flamingo");

  voters.insert(make_pair("San Diego", nameList));

  felons.push_back("Bob Buffalo");
  felons.push_back("Charles Cat");

  for_each(voters.begin(), voters.end(), &printCounty);
  cout << endl;
  auditVoterRolls(voters, felons);
  for_each(voters.begin(), voters.end(), &printCounty);

  return (0);
}
