#include <queue>
#include <stdexcept>

using namespace std;

//
// Szablon klasy ObjectPool
//
// Realizuje pul obiektw, kta moe by wykorzystywana z dowoln klas
// posiadajc domylny konstruktor.
//
// Konstruktor puli obiektw tworzy obiekt puli, ktry przekazuje obiekty
// klientom w przypadku zadania obiektu za pomoc metody acquireObject().
// Gdy klient zakoczy operacje na obiekcie, wywouje releaseObject() aby 
// zwrci obiekt do puli.
//
// Konstruktor i destruktor kadego obiektu z puli jest wywoywany tylko raz
// w czasie dziaania programu, a nie za kadym razem przy pobieraniu u zwalnianiu.
//
// Podstawowym zastosowaniem puli obiektw jest uniknicie wielokrotnego tworzenia
// i kasowania obiektw. Pula obiektw najlepiej nadaje si do aplikacji korzystajcych
// z duej liczby obiektw uywanych przez krtki czas.
//
// Dla utrzymania duej wydajnoci pula obiektw nie wykonuje kontroli poprawnoci.
// Oczekuje si, e uytkownik tylko raz zwolni kady pobrany obiekt.
// Oczekuje si, e uytkownik nie bdzie korzysta ze zwolnionego obiektu.
//
// Oczekuje si, e uytkownik nie bdzie kasowa obiektw z puli dopki wszystkie
// pobrane obiekty nie zostan zwolnione. Kasowanie puli obiektw uniewania
// wszystkie obiekty pobrane przez uytkownika, nawet jeeli ich jeszcze nie zwolni.
//
template <typename T>
class ObjectPool
{
 public:
  //
  // Tworzy pul obiektw z chunkSize obiektami.
  // Gdy w puli wyczerpi si obiekty, zostanie utworzone kolejne chunkSize
  // obiektw. Pula tylko ronie: obiekty nie s nigdy usuwane (zwalniane)
  // a do usunicia puli.
  //
  // Warto chunkSize musi by wiksza od 0.
  //
  ObjectPool(int chunkSize = kDefaultChunkSize)
    throw(invalid_argument, bad_alloc);

  //
  // Zwalnia wszystkie utworzone obiekty. Uniewania wszystkie obiekty
  // pobrane do wykorzystania.
  //
  ~ObjectPool();

  //
  // Rezerwacja obiektu do wykorzystania. Referencja tego obiektu
  // jest uniewaniana, jeeli zostanie usunita pula.
  // 
  // Klienci nie mog zwalnia obiektw!
  //
  T &acquireObject();

  //
  // Zwraca obiekt do puli. Klienci nie mog korzysta 
  // z obiektu zwrconego do puli.
  //
  void releaseObject(T &obj);

 protected:
  queue<T *> mFreeList;
  vector<T *> mAllObjects;

  int mChunkSize;
  static const int kDefaultChunkSize;

  void allocateChunk();
  static void arrayDeleteObject(T *obj);

 private:
  ObjectPool(const ObjectPool<T> &src);
  ObjectPool<T> &operator=(const ObjectPool<T> &rhs);
};
