#include <queue>
#include <vector>
#include <stdexcept>
#include <memory>

using std::queue;
using std::vector;

//
// klasa szablonowa ObjectPool
//
// Zapewnia pul obiektw, ktra moe by stosowana dla dowolnej klasy 
// zapewniajcej konstruktor domylny.
//
// Konstruktor puli obiektw tworzy obiekty, ktre przechowuje dla klientw
// a do momentu ich zadania metod acquireObject(). Gdy klient zakoczy
// korzystanie z obiektu, wywouje metod releaseObject(), aby umieci obiekt
// z powrotem w puli.
//
// Konstruktor i destruktor kadego obiektu z puli zostanie wywoany tylko jeden
// w cigu caego okresu dziaania programu. Nie bdzie wywoywany przy kadym
// pobraniu lub zwolnieniu.
//
// Gwnym zadaniem puli obiektw jest uniknicie cigego tworzenia i usuwania
// obiektw. Pula obiektw najlepiej nadaje si dla programw, ktre 
// korzystaj z wielu obiektw przez krtki okres czasu.
//
// Aby zwikszy wydajno, pula nie przeprowadza testw poprawnoci.
// Oczekuje, i uytkownik zwolni przyznany obiekt dokadnie jeden raz.
// Oczekuje, i uytkownik nie bdzie korzysta ze zwolnionego obiektu.
//
// Oczeakuje, i uytkownik nie usunie puli, dopki cho jeden obiekt zosta
// pobrany, ale nie zosta zwolniony. Usunicie puli obiektw powoduje uniewanienie
// wszystkich obiektw, ktre zostay pobrane i nie zostay zwrcone.
//
template <typename T>
class ObjectPool
{
 public:
  //
  // Tworzy pule z  chunkSize obiektami.
  // Gdy puli zabraknie obiektw, zostanie dodanych kolejne chunkSize obiektw.
  // Obiekty nie s nigdy zwalniane, poza przypadkiem niszczenia puli obiektw.
  //
  // Zgasza invalid_argument, jeli chunkSize jest <= 0.
  //
  ObjectPool(int chunkSize = kDefaultChunkSize)
    throw(std::invalid_argument, std::bad_alloc);

  //
  // Zwalnia wszystkie zaalokowane obiekty. Uniewania wszystkie pobrane i uywane
  // obiekty.
  //
  ~ObjectPool();

  //
  // Rezerwuje uycie obiektu. Referencja do obiektu staje si niewana, gdy
  // zwolniony zostanie sam obiekt puli.
  // 
  // Klientom nie wolno zwalnia obiektw!
  //
  T& acquireObject();

  //
  // Zwraca obiekt do puli. Klientom nie wolno uywa obiektw, ktre
  // zostay zwrcone do puli.
  //
  void releaseObject(T& obj);

 protected:
  //
  // mFreeList przechowuje obiekty, ktre nie s aktualnie uywane przez
  // klientw.
  //
  queue<T*> mFreeList;
  //
  // mAllObjects przechowuje wskaniki do wszystkich obiektw, uywanych lub 
  // nie. Ten wektor jest potrzebny, by zapewni poprawne zwolnienie wszystkich
  // obiektw w konstruktorze.
  //
  vector<T*> mAllObjects;

  int mChunkSize;
  static const int kDefaultChunkSize = 10;

  //
  // Alokuje mChunkSize nowych obiektw i dodaje je do mFreeList.
  //
  void allocateChunk();
  static void arrayDeleteObject(T* obj);

 private:
  // Uniemoliwia przypisywanie i przekazywanie przez warto.
  ObjectPool(const ObjectPool<T>& src);
  ObjectPool<T>& operator=(const ObjectPool<T>& rhs);
};

template<typename T>
const int ObjectPool<T>::kDefaultChunkSize;

template <typename T>
ObjectPool<T>::ObjectPool(int chunkSize) throw(std::invalid_argument,
    std::bad_alloc) : mChunkSize(chunkSize)
{
    if (mChunkSize <= 0) {
        throw std::invalid_argument("rozmiar paczki musi by dodatni");
    }
    // Tworzenie na pocztku mChunkSize obiektw.
    allocateChunk();
}

//
// Alokuje tablic mChunkSize obiektw, poniewa jest to bardziej
// wydajne ni alokacja poszczeglnych obiektw.
// Wskanik na pierwszy element tablicy zapamituje w wektorze mAllObjects
// Dodaje wskanik do kadego nowego obiektu do mFreeList.
//
template <typename T>
void ObjectPool<T>::allocateChunk()
{
    T* newObjects = new T[mChunkSize];
    mAllObjects.push_back(newObjects);
    for (int i = 0; i < mChunkSize; i++) {
        mFreeList.push(&newObjects[i]);
    }
}

//
// Funkcja zwalniania uywana w algorytmie for_each wywoywanym w destruktorze.
//
template<typename T>
void ObjectPool<T>::arrayDeleteObject(T* obj)
{
    delete [] obj;
}

template <typename T>
ObjectPool<T>::~ObjectPool()
{
    // zwolnij kad z zaalokowanych paczek
    for_each(mAllObjects.begin(), mAllObjects.end(), arrayDeleteObject);
}

template <typename T>
T& ObjectPool<T>::acquireObject()
{
    if (mFreeList.empty()) {
        allocateChunk();
    }
    T* obj = mFreeList.front();
    mFreeList.pop();
    return (*obj);
}

template <typename T>
void ObjectPool<T>::releaseObject(T& obj)
{
    mFreeList.push(&obj);
}

