#include "NameDB.h"

#include <fstream>

using namespace std;


//
// Odczytuje nazwy z pliku i wypenia baz danych.
// Baza danych jest map wic imiona z czstotliwoci ich wystpowania.
//
NameDB::NameDB(const string& nameFile) throw (invalid_argument)
{
  //
  // Otwrz plik i sprawd, czy nie pojawi si bd.
  //
  ifstream inFile(nameFile.c_str());
  if (!inFile) {
    throw invalid_argument("Nieudane otwarcie pliku.\n");
  }

  //
  // Odczytywanie kolejnych imion.
  //
  string name;
  while (inFile >> name) {
    //
    // Wyszukanie imiona w bazie danych.
    //
    if (!nameExistsAndIncrement(name)) {
      //
      // Jeli imi istnieje w bazie danych, dokonujemy
      // inkrementacji i kontynnujemy dziaanie.
      // W przeciwnym przypadku dodajemy imi do bazy z wartoci
      // licznoci wynoszc jeden.
      //
      addNewName(name);
    }
  }
  inFile.close();
}

//
// nameExistsAndIncrement
//
// Zwrca true, jeli imi istnieje w bazie danych. W przeciwnym przypadku zwraca false.
// Jeli znajdzie imi, inkrementuje jego licznik.
//
bool NameDB::nameExistsAndIncrement(const string& name)
{
  //
  // Znajdowanie imiona w obiekcie map.
  //
  map<string, int>::iterator res = mNames.find(name);
  if (res != mNames.end()) {
    res->second++;
    return (true);
  }
  return (false);
}

//
// addNewName
//
// Dodanie nowego imienia do bazy danych.
//
void NameDB::addNewName(const string& name)
{
  mNames.insert(make_pair<string, int>(name, 1));
}

//
// getNameRank
//
// Zwraca pozycj w rankingu imion.
int NameDB::getNameRank(const string& name) const
{
  int num = getAbsoluteNumber(name);

  //
  // Sprawdzenie odnalezienia imienia.
  //
  if (num == -1) {
    return (-1);
  }

  //
  // Zliczenie wszystkich imion w mapie, ktre maj wysz warto
  // licznika ni sprawdzane imi. Jeli adne imi nie ma wikszej wartoci,
  // wtedy szukane imi na ranking 1. Kade znalezione imi z wysz wartoci
  // licznika zmniejsza pozycj w rankingu o jeden.
  //
  int rank = 1;
  for (map<string, int>::const_iterator it = mNames.begin();
       it != mNames.end(); ++it) {
    if (it->second > num) {
      rank++;
    }
  }

  return (rank);
}

//
// getAbsoluteNumber
//
// Zwraca licznik zwizany z podanym imieniem.
//
int NameDB::getAbsoluteNumber(const string& name) const
{
  map<string, int>::const_iterator res = mNames.find(name);
  if (res != mNames.end()) {
    return (res->second);
  }

  return (-1);
}
