//
//  WalkLocationService.cs
//  TrackMyWalks Location Service Class (iOS)
//
//  Autor: Steven F. Daniel, 16.09.2016
//  Copyright  2016 GENIESOFT STUDIOS. All rights reserved.
//
using System;
using CoreLocation;
using TrackMyWalks.iOS;
using TrackMyWalks.Services;
using UIKit;

[assembly: Xamarin.Forms.Dependency(typeof(WalkLocationService))]
namespace TrackMyWalks.iOS
{
	// argumenty zdarzenia zawierajce szeroko i dugo geograficzn
	public class Coordinates : EventArgs, IWalkCoordinates
	{
		public double latitude { get; set; }
		public double longitude { get; set; }
	}

	// klasa usugi lokalizacji dziedziczca po 
	// interfejsie IWalkLocationService
	public class WalkLocationService : IWalkLocationService
	{
		// deklaracja menedera lokalizacji
		CLLocationManager locationManager;
		CLLocation newLocation;

		// definicja delegatu EventHandler wywoywanego
// w odpowiedzi na uzyskanie danych dotyczcych lokalizacji
		public event EventHandler<IWalkCoordinates> MyLocation;

		// metoda rozpoczynajca sprawdzanie lokalizacji
		public void GetMyLocation()
		{
			locationManager = new CLLocationManager();

			// sprawdzenie czy usugi lokalizacji s wczone
			if (CLLocationManager.LocationServicesEnabled)
			{
				// ustawienie dokadnoci w metrach
				locationManager.DesiredAccuracy = 1;

				// iOS 8 ma dodatkowe wymagania dotyczce uprawnie
				if (UIDevice.CurrentDevice.CheckSystemVersion(8, 0))
				{
					// wykonywanie zmian lokalizacji w tle
					locationManager.RequestAlwaysAuthorization();
				}

				// iOS 9 zawiera now metod pozwalajc na odbieranie
				// danych lokalizacyjnych w tle, gdy aplikacja jest zawieszona
				if (UIDevice.CurrentDevice.CheckSystemVersion(9, 0))
				{
					locationManager.AllowsBackgroundLocationUpdates = true;
				}

				// metody CLLocationManagerDelegate

				// wywoywana przy kadej zmianie lokalizacji
				locationManager.LocationsUpdated += (object sender, CLLocationsUpdatedEventArgs e) =>
				{
					locationUpdated(e);
				};

				// to zdarzenie jest wywoywane, gdy zostanie wykryta zmiana, tzn. gdy uytkownik
				// wyczy usugi lokalizacji
				locationManager.AuthorizationChanged += (object sender, CLAuthorizationChangedEventArgs e) =>
				{
					didAuthorizationChange(e);

					// wykonywanie zmian lokalizacji na pierwszym planie
					locationManager.RequestWhenInUseAuthorization();
				};
			}
		}

		// metoda wywoywana po zmianie lokalizacji
		public void locationUpdated(CLLocationsUpdatedEventArgs e)
		{
			// utworzenie wsprzdnych lokalizacji 
			var coords = new Coordinates();

			// pobranie listy znalezionych lokalizacji
			var locations = e.Locations;

			// pobranie dugoci i szerokoci geograficznej z tablicy lokalizacji
			coords.latitude = locations[locations.Length - 1].Coordinate.Latitude;
			coords.longitude = locations[locations.Length - 1].Coordinate.Longitude;

			// konwersja obu wartoci na obiekt CLLocation
			newLocation = new CLLocation(coords.latitude, coords.longitude);
			MyLocation(this, coords);
		}

		public void didAuthorizationChange(CLAuthorizationChangedEventArgs authStatus)
		{
			switch (authStatus.Status)
			{
				case CLAuthorizationStatus.AuthorizedAlways:
					locationManager.RequestAlwaysAuthorization();
					break;
				case CLAuthorizationStatus.AuthorizedWhenInUse:
					locationManager.StartUpdatingLocation();
					break;
				case CLAuthorizationStatus.Denied:
					UIAlertView alert = new UIAlertView();

					alert.Title = "Usugi lokalizacji s wyczone";
					alert.AddButton("OK");
					alert.AddButton("Anuluj");
					alert.Message = "Wcz lokalizacj dla tej aplikacji w\n Ustawieniach swojego iPhone'a.";
					alert.AlertViewStyle = UIAlertViewStyle.Default;

					alert.Show();

					alert.Clicked += (object s, UIButtonEventArgs ev) =>
					{
						var Button = ev.ButtonIndex;
					};
					break;
				default:
					break;
			}
		}

		// oblicza odlego midzy dwoma punktami
		public double GetDistanceTravelled(double lat, double lon)
		{
			// oblicza dystans przebyty od biecej do poprzedniej lokalizacji
			var distance = newLocation.DistanceFrom(new CLLocation(lat, lon)) / 1000;
			return distance;
		}

		// zatrzymuje aktualizowanie lokalizacji, gdy obiekt zostanie ustawiony na null
		~WalkLocationService()
		{
			locationManager.StopUpdatingLocation();
		}
	}
}