#include <array>
#include <vector>
#include <string>
#include <memory>
#include <gtest/gtest.h>
#include "scoped_timer.h"


//
// Ten kod pokazuje, że można podzielić tablicę
// dużych obiektów, aby uzyskać wiele równoległych
// tablic mniejszych obiektów.
//
// Typ OriginalUser reprezentuje strukturę User struct sprzed podziału
// jej na mniejsze fragmenty i rozpoczęcia korzystania z równoległych tablic.
//

// Ta struktura w książce nosi nazwę User.
// Tu nazwałem ją OriginalUser, aby odróżnić od zdefiniowanej poniżej
// mniejszej wersji struktury User .
struct OriginalUser {
  std::string name_;
  std::string username_;
  std::string password_;
  std::string security_question_;
  std::string security_answer_;
  short level_{};
  bool is_playing_{};
};

auto num_users_at_level(short level, const std::vector<OriginalUser>& users) {
  ScopedTimer t{"num_users_at_level (using OriginalUser)"};
  auto num_users = 0;
  for (const auto& user : users)
    if (user.level_ == level)
      ++num_users;
  return num_users;
}

auto num_playing_users(const std::vector<OriginalUser>& users) {
  ScopedTimer t{"num_playing_users (using OriginalUser)"};
  return std::count_if(
    users.begin(),
    users.end(),
    [](const auto& user) {
      return user.is_playing_;
    });
}


//
// Poniżej znajdują się fragmenty struktury OriginalUser po 
// jej podziale na mniejsze struktury w celu szybszego przetwarzania.
//

struct AuthInfo {
  std::string username_;
  std::string password_;
  std::string security_question_;
  std::string security_answer_;
};

struct User {
  std::string name_;
  std::unique_ptr<AuthInfo> auth_info_;
  short level_{};
  bool is_playing_{};
};


auto num_users_at_level(short level, const std::vector<User>& users) {
  ScopedTimer t{"num_users_at_level (using User)"};
  auto num_users = 0;
  for (const auto& user : users) {
    if (user.level_ == level) {
      ++num_users;
    }
  }
  return num_users;
}

auto num_playing_users(const std::vector<User>& users) {
  ScopedTimer t{"num_playing_users (using User)"};
  return std::count_if(
    users.begin(),
    users.end(),
    [](const auto& user) {
      return user.is_playing_;
    });
}

//
// Oto funkcje do obliczania wartości po pobraniu
// składowych level_ i is_playing_ ze struktury User.
// Jest to najszybszy sposób na uzyskanie wyników.
//

auto num_users_at_level(short level, const std::vector<short>& users) {
  ScopedTimer t{"num_users_at_level using vector<short>"};
  return std::count(users.begin(), users.end(), level);
}

auto num_playing_users(const std::vector<bool>& users) {
  ScopedTimer t{"num_playing_users using vector<bool>"};
  return std::count(users.begin(), users.end(), true);
}

//
// Funkcje narzędziowe do generowania danych testowych
//
auto gen_is_playing() {
  return static_cast<bool>(std::rand() % 2);
}

auto gen_level() {
  return static_cast<short>(std::rand() % 100);
}
auto create_users(size_t count) {
  auto vec = std::vector<User>(count);
  for (auto& user : vec) {
    user.name_ = "some name";
    user.level_ = gen_level();
    user.is_playing_ = gen_is_playing();
  }
  return vec;
}

auto create_levels(size_t count) {
  auto vec = std::vector<short>(count);
  for (auto& level : vec) {
    level = std::rand() % 100;
  }
  return vec;
};

auto create_playing_users(size_t count) {
  auto vec = std::vector<bool>(count); // Zauważ, że typ std::vector<bool> to przypadek specjalny
  for (auto&& is_playing : vec) {
    is_playing = static_cast<bool>(std::rand() % 2);
  }
  return vec;
};

TEST(ParallelArrays, CompareProcessingTime) {
  std::cout << "sizeof(OriginalUser): " << sizeof(OriginalUser) << " w bajtach" << '\n';
  std::cout << "sizeof(User): " << sizeof(User) << " w bajtach" << '\n';

  auto num_objects = 1'000'000; // Zwiększ tę wartość, jeśli chcesz utworzyć więcej obiektów

  std::cout << "Tworzenie obiektów...." << '\n';
  auto original_users = std::vector<OriginalUser>(num_objects);
  auto users = create_users(num_objects);
  auto user_levels = create_levels(num_objects);
  auto playing_users = create_playing_users(num_objects);

  std::cout << "gotowe." << '\n';

  auto n = 0ul;
  auto level = short{5};

  std::cout << '\n' << "+++ Obliczanie wyników z użyciem typu OriginalUser +++" << '\n';

  n += num_users_at_level(level, original_users);
  n += num_users_at_level(level, original_users);
  n += num_users_at_level(level, original_users);

  n += num_playing_users(original_users);
  n += num_playing_users(original_users);
  n += num_playing_users(original_users);

  std::cout << '\n' << "+++ Obliczanie wyników z użyciem typu User +++" << '\n';
  n += num_users_at_level(level, users);
  n += num_users_at_level(level, users);
  n += num_users_at_level(level, users);

  n += num_playing_users(users);
  n += num_playing_users(users);
  n += num_playing_users(users);

  std::cout << '\n' << "+++ Obliczanie wyników z użyciem typów vector<short> i vector<bool> +++" << '\n';
  n += num_users_at_level(level, user_levels);
  n += num_users_at_level(level, user_levels);
  n += num_users_at_level(level, user_levels);

  n += num_playing_users(playing_users);
  n += num_playing_users(playing_users);
  n += num_playing_users(playing_users);

  // Wyświetlanie n, aby uniknąć usunięcia zmiennej w wyniku optymalizacji...
  std::cout << n << '\n';
}
