#include <cassert>
#include <vector>
#include <gtest/gtest.h>

//
// Ten kod pokazuje, jak używanie słowa const 
// pomaga rozróżniać funkcje modyfikujące od
// niemodyfikujących.
//

class Person {
public:
  auto age() const {
    return age_;
  }
  auto set_age(int age) {
    age_ = age;
  }

private:
  int age_{};
};


class Team {
public:
  auto& leader() const {
    return leader_;
  }
  auto& leader() {
    return leader_;
  }

private:
  Person leader_{};
};


// Ta funkcja nie modyfikuje zmiennej teams,
// ponieważ zmienną teams zadeklarowano z użyciem const.
// Próba modyfikacji zmiennej teams spowoduje błąd kompilacji.
auto nonmutating_func(const std::vector<Team>& teams) {
  auto tot_age = int{0};

  // Kod się skompiluje. leader() and age() są zadeklarowane z modyfikatorem const.
  for (const auto& team: teams)
    tot_age += team.leader().age();

  // Ten fragment się nie skompiluje. Funkcja set_age() wymaga modyfikowalnego obiektu.
  //  for(auto& team: teams)
  //    team.leader().set_age(20);
}


// Ta funkcja może modyfikować obiekty typu Team.
auto mutating_func(std::vector<Team>& teams) {
  auto tot_age = int{0};

  // Kod się skompiluje. Funkcje z modyfikatorem const można wywoływać dla 
  // modyfikowalnych obiektów.
  for (const auto& team: teams)
    tot_age += team.leader().age();

  // Kod się skompiluje. teams jest modyfikowalną zmienną.
  for (auto& team: teams)
    team.leader().set_age(20);
}

TEST(ConstCorrectness, Nonmutating) {
  const auto teams = std::vector<Team>{};
  nonmutating_func(teams);
  // mutating_func(teams); // Nie skompiluje się, wektor teams zadeklarowano z modyfikatorem const.
}

TEST(ConstCorrectness, Mutating) {
  auto teams = std::vector<Team>{};
  nonmutating_func(teams);
  mutating_func(teams);
}

