#include "uart_dma.h"

#define UART2EN			(1U<<17)
#define GPIOAEN			(1U<<0)

#define CR1_TE			(1U<<3)
#define CR1_RE			(1U<<2)
#define CR1_UE			(1U<<13)
#define SR_TXE			(1U<<7)

#define CR3_DMAT		(1U<<7)
#define CR3_DMAR		(1U<<6)
#define SR_TC			(1U<<6)
#define CR1_TCIE		(1U<<6)

#define UART_BAUDRATE	115200
#define CLK				16000000

#define DMA1EN			    (1U<<21)
#define DMA_SCR_EN  		(1U<<0)
#define DMA_SCR_MINC		(1U<<10)
#define DMA_SCR_PINC		(1U<<9)
#define DMA_SCR_CIRC		(1U<<8)
#define DMA_SCR_TCIE		(1U<<4)
#define DMA_SCR_TEIE		(1U<<2)
#define DMA_SFCR_DMDIS		(1U<<2)

#define HIFCR_CDMEIF5		(1U<<8)
#define HIFCR_CTEIF5		(1U<<9)
#define HIFCR_CTCIF5		(1U<<11)

#define HIFCR_CDMEIF6		(1U<<18)
#define HIFCR_CTEIF6		(1U<<19)
#define HIFCR_CTCIF6		(1U<<21)

#define HIFSR_TCIF5		(1U<<11)
#define HIFSR_TCIF6		(1U<<21)

static uint16_t compute_uart_bd(uint32_t periph_clk, uint32_t baudrate);
static void uart_set_baudrate(uint32_t periph_clk, uint32_t baudrate);

char uart_data_buffer[UART_DATA_BUFF_SIZE];

uint8_t g_rx_cmplt;
uint8_t g_tx_cmplt;

uint8_t g_uart_cmplt;

void uart2_rx_tx_init(void)
{
	/************* Konfigurowanie pinów GPIO dla układu UART ********************/
	/* 1. Włączanie dostępu do zegara dla GPIOA*/
	RCC->AHB1ENR |= GPIOAEN;

	/* 2. Ustawianie PA2 w tryb funkcji alternatywnej */
	GPIOA->MODER &= ~(1U<<4);
	GPIOA->MODER |=	 (1U<<5);

	/* 3. Ustawianie PA3 w tryb funkcji alternatywnej */
	GPIOA->MODER &= ~(1U<<6);
	GPIOA->MODER |=	 (1U<<7);

	/* 4. Ustawianie typu funkcji alternatywnej PA2 na AF7(UART2_TX)*/
	GPIOA->AFR[0] |= (1U<<8);
	GPIOA->AFR[0] |= (1U<<9);
	GPIOA->AFR[0] |= (1U<<10);
	GPIOA->AFR[0] &= ~(1U<<11);

	/* 5. Ustawianie typu funkcji alternatywnej PA3 na AF7(UART2_TX)*/
	GPIOA->AFR[0] |= (1U<<12);
	GPIOA->AFR[0] |= (1U<<13);
	GPIOA->AFR[0] |= (1U<<14);
	GPIOA->AFR[0] &= ~(1U<<15);
	
	/************* Konfigurowanie modułu UART ********************/
	/* 6. Włączanie dostępu do zegara dla UART2*/
	RCC->APB1ENR |= UART2EN;

	/* 7. Ustawianie prędkości transmisji */
	uart_set_baudrate(CLK,UART_BAUDRATE);

	/* 8. Wybór trybu DMA dla transmisji i odbioru */
	USART2->CR3 = CR3_DMAT |CR3_DMAR;

	/* 9. Ustawianie kierunku transferu */
	 USART2->CR1 = CR1_TE |CR1_RE;

	/* 10. Zerowanie flagi TC */
	 USART2->SR &=~SR_TC;

	/* 11. Włączanie TCIE */
	 USART2->CR1 |=CR1_TCIE;

	/* 12. Włączanie modułu UART */
	 USART2->CR1 |= CR1_UE;

	/*13. Włączanie przerwania USART2 w kontrolerze NVIC*/
	NVIC_EnableIRQ(USART2_IRQn);

}


void dma1_init(void)
{
   /* Włączanie dostępu do zegara dla DMA*/
	RCC->AHB1ENR |=DMA1EN;

	/* Włączanie przerwania 6. strumienia DMA w kontrolerze NVIC*/
	NVIC_EnableIRQ(DMA1_Stream6_IRQn);
}


void dma1_stream5_uart_rx_config(void)
{
	/* Wyłączanie strumienia DMA */
	DMA1_Stream5->CR &=~DMA_SCR_EN;

	/* Czekanie na wyłączenie strumienia DMA */
	while((DMA1_Stream5->CR & DMA_SCR_EN)){}

	/* Resetowanie flag przerwań dla strumienia 5. */
	DMA1->HIFCR = HIFCR_CDMEIF5 |HIFCR_CTEIF5|HIFCR_CTCIF5;

	/* Ustawianie adresu układu peryferyjnego */
	DMA1_Stream5->PAR = (uint32_t)(&(USART2->DR));

	/* Ustawianie adresu w pamięci */
	DMA1_Stream5->M0AR = (uint32_t)(&uart_data_buffer);

	/* Ustawianie liczby transferów */
	DMA1_Stream5->NDTR = (uint16_t)UART_DATA_BUFF_SIZE;

	/* Wybieranie kanału 4. */
	DMA1_Stream5->CR &= ~(1u<<25);
	DMA1_Stream5->CR &= ~(1u<<26);
	DMA1_Stream5->CR |= (1u<<27);

	/* Włączanie inkrementacji adresu pamięci */
	DMA1_Stream5->CR |=DMA_SCR_MINC;

	/* Włączanie przerwania przy zakończeniu transferu */
	DMA1_Stream5->CR |= DMA_SCR_TCIE;

	/* Włączanie trybu kołowego */
	DMA1_Stream5->CR |=DMA_SCR_CIRC;

	/* Ustawianie kierunku transferu: od układu peryferyjnego do pamięci */
	DMA1_Stream5->CR &=~(1U<<6);
	DMA1_Stream5->CR &=~(1U<<7);

	/* Włączanie strumienia DMA */
	DMA1_Stream5->CR |= DMA_SCR_EN;

	/* Włączanie przerwania 5. strumienia DMA w kontrolerze NVIC*/
	NVIC_EnableIRQ(DMA1_Stream5_IRQn);

}

void dma1_stream6_uart_tx_config(uint32_t msg_to_snd, uint32_t msg_len)
{
	/* Wyłączanie strumienia DMA */
	DMA1_Stream6->CR &=~DMA_SCR_EN;

	/* Czekanie na wyłączenie strumienia DMA */
	while((DMA1_Stream6->CR & DMA_SCR_EN)){}

	/* Resetowanie flag przerwań dla strumienia 6 */
	DMA1->HIFCR = HIFCR_CDMEIF6 |HIFCR_CTEIF6|HIFCR_CTCIF6;

	/* Ustawianie adresu układu peryferyjnego */
	DMA1_Stream6->PAR = (uint32_t)(&(USART2->DR));

	/* Ustawianie adresu pamięci */
	DMA1_Stream6->M0AR = msg_to_snd;

	/* Ustawianie liczby transferów */
	DMA1_Stream6->NDTR = msg_len;

	/* Wybieranie kanału 4. */
	DMA1_Stream6->CR &= ~(1u<<25);
	DMA1_Stream6->CR &= ~(1u<<26);
	DMA1_Stream6->CR |= (1u<<27);

	/* Włączanie inkrementacji adresu pamięci */
	DMA1_Stream6->CR |=DMA_SCR_MINC;

	/* Ustawianie kierunku transferu: od pamięci do układu peryferyjnego */
	DMA1_Stream6->CR |=(1U<<6);
	DMA1_Stream6->CR &=~(1U<<7);

	/* Ustawianie przerwania przy zakończeniu transferu */
	DMA1_Stream6->CR |= DMA_SCR_TCIE;

	/* Włączanie strumienia DMA */
	DMA1_Stream6->CR |= DMA_SCR_EN;

}

static uint16_t compute_uart_bd(uint32_t periph_clk, uint32_t baudrate)
{
	return ((periph_clk +( baudrate/2U ))/baudrate);
}


static void uart_set_baudrate(uint32_t periph_clk, uint32_t baudrate)
{
	USART2->BRR  = compute_uart_bd(periph_clk,baudrate);
}


void DMA1_Stream6_IRQHandler(void)
{
	if((DMA1->HISR) & HIFSR_TCIF6)
	{
		// tu można coś zrobić
		g_tx_cmplt = 1;
		/* Resetowanie flagi */
		DMA1->HIFCR |= HIFCR_CTCIF6;
	}
}


void DMA1_Stream5_IRQHandler(void)
{
	if((DMA1->HISR) & HIFSR_TCIF5)
	{

		g_rx_cmplt = 1;


		/* Resetowanie flagi */
		DMA1->HIFCR |= HIFCR_CTCIF5;
	}
}

void USART2_IRQHandler(void)
{
	g_uart_cmplt  = 1;

	/* Resetowanie flagi przerwania TC */
	USART2->SR &=~SR_TC;
}


