const bcrypt = require('bcrypt');
const jwt = require('jsonwebtoken');
const {
  AuthenticationError,
  ForbiddenError
} = require('apollo-server-express');
const mongoose = require('mongoose');
require('dotenv').config();

const gravatar = require('../util/gravatar');

module.exports = {
  newNote: async (parent, args, { models, user }) => {
    if (!user) {
      throw new AuthenticationError('Tylko zalogowani użytkownicy mogą tworzyć notatki.');
    }

    return await models.Note.create({
      content: args.content,
      author: mongoose.Types.ObjectId(user.id),
      favoriteCount: 0
    });
  },
  deleteNote: async (parent, { id }, { models, user }) => {
    // Jeżeli użytkownik nie zostanie znaleziony, należy zgłosić błąd uwierzytelniania.
    if (!user) {
      throw new AuthenticationError('Tylko zalogowani użytkownicy mogą usuwać notatki.');
    }

    // Odszukanie notatki.
    const note = await models.Note.findById(id);
    // Jeżeli właściciel notatki nie jest bieżącym użytkownikiem, należy zgłosić błąd.
    if (note && String(note.author) !== user.id) {
      throw new ForbiddenError("Nie masz uprawnień do usunięcia notatki.");
    }

    try {
      // Jeżeli wszystko się zgadza, należy usunąć notatkę.
      await note.remove();
      return true;
    } catch (err) {
      // W przypadku wystąpienia błędu, wartością zwrotną będzie false.
      return false;
    }
  },
  updateNote: async (parent, { content, id }, { models, user }) => {
    // Jeżeli użytkownik nie zostanie znaleziony, należy zgłosić błąd uwierzytelniania.
    if (!user) {
      throw new AuthenticationError('Tylko zalogowani użytkownicy mogą uaktualniać notatki.');
    }

    // Odszukanie notatki.
    const note = await models.Note.findById(id);
    // Jeżeli właściciel notatki nie jest bieżącym użytkownikiem, należy zgłosić błąd.
    if (note && String(note.author) !== user.id) {
      throw new ForbiddenError("Nie masz uprawnień do uaktualnienia notatki.");
    }

    // Uaktualnienie notatki w bazie danych i zwrot uaktualnionej notatki.
    return await models.Note.findOneAndUpdate(
      {
        _id: id
      },
      {
        $set: {
          content
        }
      },
      {
        new: true
      }
    );
  },
  toggleFavorite: async (parent, { id }, { models, user }) => {
    // Jeżeli użytkownik nie zostanie znaleziony, należy zgłosić błąd uwierzytelniania.
    if (!user) {
      throw new AuthenticationError();
    }

    // Sprawdzenie, czy użytkownik oznaczył już daną notatkę jako ulubioną.
    let noteCheck = await models.Note.findById(id);
    const hasUser = noteCheck.favoritedBy.indexOf(user.id);

    // Jeżeli nazwa użytkownika znajduje się na liście, należy ją
    // z niej usunąć i zmniejszyć o 1 wartość właściwości favoriteCount.
    if (hasUser >= 0) {
      return await models.Note.findByIdAndUpdate(
        id,
        {
          $pull: {
            favoritedBy: mongoose.Types.ObjectId(user.id)
          },
          $inc: {
            favoriteCount: -1
          }
        },
        {
          // Właściwości new należy przypisać wartość true, aby zwrócić uaktualnioną notatkę.
          new: true
        }
      );
    } else {
      // Jeżeli nazwa użytkownika nie znajduje się na liście, należy ją
      // dodać do listy i zwiększyć o 1 wartość właściwości favoriteCount.
      return await models.Note.findByIdAndUpdate(
        id,
        {
          $push: {
            favoritedBy: mongoose.Types.ObjectId(user.id)
          },
          $inc: {
            favoriteCount: 1
          }
        },
        {
          new: true
        }
      );
    }
  },
  signUp: async (parent, { username, email, password }, { models }) => {
    // Normalizacja adresu e-mail.
    email = email.trim().toLowerCase();
    // Utworzenie wartości hash na podstawie hasła.
    const hashed = await bcrypt.hash(password, 10);
    // Utworzenie adresu URL gravatar.
    const avatar = gravatar(email);
    try {
      const user = await models.User.create({
        username,
        email,
        avatar,
        password: hashed
      });

      // Utworzenie i zwrot tokena JSON Web.
      return jwt.sign({ id: user._id }, process.env.JWT_SECRET);
    } catch (err) {
      // W przypadku wystąpienia problemu podczas tworzenia konta, należy zgłosić błąd.
      throw new Error('Błąd podczas tworzenia konta.');
    }
  },

  signIn: async (parent, { username, email, password }, { models }) => {
    if (email) {
      // Normalizacja adresu e-mail.
      email = email.trim().toLowerCase();
    }
    const user = await models.User.findOne({
      $or: [{ email }, { username }]
    });

    // Jeżeli użytkownik nie zostanie znaleziony, należy zgłosić błąd uwierzytelniania.
    if (!user) {
      throw new AuthenticationError('Błąd podczas uwierzytelniania.');
    }

    // Jeżeli hasła są różne, należy zgłosić błąd uwierzytelniania.
    const valid = await bcrypt.compare(password, user.password);
    if (!valid) {
      throw new AuthenticationError('Błąd podczas uwierzytelniania.');
    }

    // Utworzenie i zwrot tokena JSON Web.
    return jwt.sign({ id: user._id }, process.env.JWT_SECRET);
  }
};
