const express = require('express');
const { ApolloServer } = require('apollo-server-express');
const jwt = require('jsonwebtoken');
const helmet = require('helmet');
const cors = require('cors');
const depthLimit = require('graphql-depth-limit');
const { createComplexityLimitRule } = require('graphql-validation-complexity');
require('dotenv').config();

const db = require('./db');
const models = require('./models');
const typeDefs = require('./schema');
const resolvers = require('./resolvers');

// Uruchomienie serwera nasłuchującego na porcie wskazanym w pliku .env, czyli na porcie 4000.
const port = process.env.PORT || 4000;
const DB_HOST = process.env.DB_HOST;

const app = express();

db.connect(DB_HOST);

// Oprogramowanie pośredniczące związane z zapewnieniem bezpieczeństwa.
app.use(helmet());
// Oprogramowanie pośredniczące związane z CORS.
app.use(cors());

// Pobranie z tokena JWT informacji o użytkowniku.
const getUser = token => {
  if (token) {
    try {
      // Zwrot z tokena informacji o użytkowniku.
      return jwt.verify(token, process.env.JWT_SECRET);
    } catch (err) {
      // W przypadku problemu z tokenem, należy wyświetlić komunikat błędu.
      throw new Error('Nieprawidłowa sesja.');
    }
  }
};

// Konfiguracja serwera Apollo.
// Uaktualniona o element `validationRules`.
const server = new ApolloServer({
  typeDefs,
  resolvers,
  validationRules: [depthLimit(5), createComplexityLimitRule(1000)],
  context: ({ req }) => {
    // Pobranie z nagłówków tokena użytkownika.
    const token = req.headers.authorization;
    // Próba pobrania z tokena informacji o użytkowniku.
    const user = getUser(token);
    // Dodanie do kontekstu modeli bazy danych i użytkownika.
    return { models, user };
  }
});

// Zastosowanie oprogramowania pośredniczącego Apollo GraphQL i zdefiniowanie ścieżki dostępu do /api.
server.applyMiddleware({ app, path: '/api' });

app.listen({ port }, () =>
  console.log(
    `Serwer GraphQL działa pod adresem http://localhost:${port}${server.graphqlPath}`
  )
);
