#ifndef PICTURE_H
#define PICTURE_H

#include "lodepng.h"
#include <string>
#include <vector>

using namespace std;

class Picture
{
 public:
   /**
      Tworzy obraz o zerowej szerokości i wysokości.
   */
   Picture();

   /**
      Tworzy obraz z danego pliku PNG.
      @param filename nazwa pliku określająca obraz PNG.
   */
   Picture(string filename);

   /**
      Tworzy obraz o jednokolorowych pikselach (domyślnie
      białych).
      @param red wartość koloru czerwonego pikseli (od 0 do 255)
      @param green wartość koloru zielonego pikseli (od 0 do 255)
      @param blue wartość koloru niebieskiego pikseli (od 0 do 255)
      @param width szerokość obrazu
      @param height wysokość obrazu
   */
   Picture(int width, int height, int red = 255, int green = 255, int blue = 255);

   /**
      Tworzy obraz na podstawie dwuwymiarowego wektora poziomów szarości.
      @param grays poziomy szarości
   */
   Picture(const vector<vector<int> >& grays);

   /**
      Zwraca szerokość obrazu.
      @return szerokość
   */   
   int width() const { return _width; }

   /**
      Zwraca wysokość obrazu.
      @return wysokość
   */
   int height() const { return _height; }

   /**
      Zapisuje obraz w podanym pliku.
      @param filename nazwa pliku określająca obraz PNG.
   */
   void save(string filename) const;

   /**
      Zwraca wartość koloru czerwonego na danej pozycji.
      @param x współrzędna x (w pionie)
      @param y współrzędna y (w poziomie)
      @return wartość koloru czerwonego piksela (od 0 do 255)
      albo 0, jeśli podany punkt nie należy do obrazu.
   */
   int red(int x, int y) const;

   /**
      Zwraca wartość koloru zielonego na danej pozycji.
      @param x współrzędna x (w pionie)
      @param y współrzędna y (w poziomie)
      @return wartość koloru zielonego piksela (od 0 do 255)
      albo 0, jeśli podany punkt nie należy do obrazu.
   */
   int green(int x, int y) const;

   /**
      Zwraca wartość koloru niebieskiego na danej pozycji.
      @param x współrzędna x (w pionie)
      @param y współrzędna y (w poziomie)
      @return wartość koloru niebieskiego piksela (od 0 do 255)
      albo 0, jeśli podany punkt nie należy do obrazu.
   */   
   int blue(int x, int y) const;

   /**
      Nadaje pikselowi podany kolor, w razie potrzeby
      rozszerza obraz.
      @param x współrzędna x (w pionie)
      @param y współrzędna y (w poziomie)
      @param red wartość koloru czerwonego piksela (od 0 do 255)
      @param green wartość koloru zielonego piksela (od 0 do 255)
      @param blue wartość koloru niebieskiego piksela (od 0 do 255)
   */   
   void set(int x, int y, int red, int green, int blue);

   /**
      Zwraca poziomy szarości wszystkich pikseli obrazu.
      @return dwuwymiarowa tablica wartości szarości (od 0 do 255)
   */
   vector<vector<int> > grays() const;

   /**
      Dodaje do obrazu na podanej pozycji drugi, w razie potrzeby
      obraz jest rozszerzany.
      @param drugi dodawany obraz
      @param x współrzędna x (kolumna) górnego lewego rogu
      @param y współrzędna y (wiersz) górnego lewego rogu
   */
   void add(const Picture& other, int x = 0, int y = 0);

 private:
   void ensure(int x, int y);
   
   vector<unsigned char> _values;
   int _width;
   int _height;   
};

#endif
