﻿Function Get-OptimalSize
{
 <#
  .Synopsis
    Przekształca bajty w odpowiednią jednostkę miary.
   .Description
    Funkcja Get-OptimalSize przekształca bajty w odpowiednią jednostkę 
    miary. Zwraca łańcuchową reprezentację liczby.
   .Example
    Get-OptimalSize 1025
    Przekształca 1025 bajtów na 1,00 kilobajt
    .Example
    Get-OptimalSize -sizeInBytes 10099999 
    Przekształca 10099999 bajtów na 9,63 megabajta
   .Parameter SizeInBytes
    Rozmiar w bajtach do konwersji
   .Inputs
    [int64]
   .OutPuts
    [string]
   .Notes
    NAZWA:  Get-OptimalSize
    AUTOR: Ed Wilson
    OSTATNIA EDYCJA: 30.6.2012
    SŁOWA KLUCZOWE: techniki pisania skryptów, moduły
   .Link
     Http://www.ScriptingGuys.com
 #Requires -Version 2.0
 #>
[CmdletBinding()]
param(
      [Parameter(Mandatory = $true,Position = 0,valueFromPipeline=$true)]
      [int64]
      $sizeInBytes
) #end param
 Switch ($sizeInBytes) 
  {
   {$sizeInBytes -ge 1TB} {"{0:n2}" -f  ($sizeInBytes/1TB) + " terabajtów";break}
   {$sizeInBytes -ge 1GB} {"{0:n2}" -f  ($sizeInBytes/1GB) + " gigabajtów";break}
   {$sizeInBytes -ge 1MB} {"{0:n2}" -f  ($sizeInBytes/1MB) + " megabajtów";break}
   {$sizeInBytes -ge 1KB} {"{0:n2}" -f  ($sizeInBytes/1KB) + " kilobajtów";break}
   Default { "{0:n2}" -f $sizeInBytes + " bajtów" }
  } #end switch
  $sizeInBytes = $null
} #end Function Get-OptimalSize 

Function Get-ComputerInfo
{
 <#
  .Synopsis
    Pobiera podstawowe informacje o komputerze. 
   .Description
    Polecenie Get-ComputerInfo pobiera podstawowe informacje, takie jak
    nazwa komputera, nazwa domeny oraz aktualnie zalogowany użytkownik
    na lokalnym lub zdalnym komputerze.
   .Example
    Get-ComputerInfo 
    Zwraca nazwę komputera, nazwę domeny i aktualnie zalogowanego użytkownika
    z komputera lokalnego.
    .Example
    Get-ComputerInfo -computer berlin
    Zwraca nazwę komputera, nazwę domeny oraz aktualnie zalogowanego użytkownika
    ze zdalnego komputera o nazwie berlin.
   .Parameter Computer
    Nazwa zdalnego komputera, z którego mają zostać pobrane informacje
   .Inputs
    [string]
   .OutPuts
    [object]
   .Notes
    NAZWA:  Get-ComputerInfo
    AUTOR: Ed Wilson
    OSTATNIA EDYCJA: 30.6.2012
    SŁOWA KLUCZOWE: zarządzanie pulpitem, podstawowe informacje
   .Link
     Http://www.ScriptingGuys.com
 #Requires -Version 2.0
 #>
 Param([string]$computer=$env:COMPUTERNAME)
 $wmi = Get-WmiObject -Class win32_computersystem -ComputerName $computer
 $pcinfo = New-Object psobject -Property @{"host" = $wmi.DNSHostname
           "domain" = $wmi.Domain 
           "user" = $wmi.Username}
 $pcInfo
} #end function Get-ComputerInfo