﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;
using Microsoft.Win32;

namespace ColorSelectorAddIn
{
    [Guid("62175059-FD7E-407a-9EF3-5D07F2B704E8")]
    public partial class PaletteControl : UserControl
    {
        #region Pola

        string _code = "";
        bool _generateVB = false;

        #endregion

        #region Właściwości

        public string Code
        {
            get { return _code; }
        }

        public bool GenerateVB
        {
            get { return _generateVB; }
        }

        #endregion

        #region Konstruktor

        public PaletteControl()
        {
            InitializeComponent();

            // Zdarzenie informujące o przesuwaniu myszy nad 
            // bitmapą palety kolorów.
            this.pictureBoxPalette.MouseMove += new MouseEventHandler(pictureBoxPalette_MouseMove);

            // Zmiana kursora nad bitmapą.
            this.pictureBoxPalette.Cursor = System.Windows.Forms.Cursors.Cross;
        }

        #endregion

        #region Metody obsługi zdarzeń i delegaty


        void pictureBoxPalette_MouseMove(object sender, MouseEventArgs e)
        {
            // Pobiera kolor odpowiadający bieżącej pozycji kursora.
            Color color = GetPointColor(e.X, e.Y);

            // Aktualizuje na podstawie pobranego koloru etykiety z wartościami składowych RGB 
            // oraz drugie pole z rysunkiem .
            DisplayColor(color);

            // Generuje kod struktury Color w języku 
            // Visual Basic lub C#. 
            SetCode(color, _generateVB);
        }

        public event EventHandler ColorSelected;

        protected virtual void OnColorSelected(EventArgs e)
        {
            if (ColorSelected != null)
                ColorSelected(this, e);
        }

        private void pictureBoxPalette_Click(object sender, EventArgs e)
        {
            OnColorSelected(new EventArgs());
        }

        #endregion

        #region Podprogramy prywatne

        /// <summary>
        /// Zwraca strukturę Color reprezentującą kolor 
        /// piksela o określonych współrzędnych x i y.
        /// </summary>
        /// <param name="x">Współrzędna 'x' kursora.</param>
        /// <param name="y">Współrzędna 'y' kursora.</param>
        /// <returns>Struktura Color odpowiadająca kolorowi pod kursorem.</returns>
        private Color GetPointColor(int x, int y)
        {
            // Pobiera bitmapę z pola zawierającego rysunek palety.
            Bitmap bmp = (Bitmap)pictureBoxPalette.Image;

            // Używa metody GetPixel do pobrania struktury
            // Color odpowiadającej bieżącej pozycji kursora.
            Color color = bmp.GetPixel(x, y);

            // Zwraca strukturę Color.
            return color;
        }

        /// <summary>
        /// Wyświetla wartości składowych RGB danego koloru. Ponadto ustawia
        /// kolor tła drugiego pola z rysunkiem.
        /// </summary>
        /// <param name="color">Wyświetlany kolor</param>
        private void DisplayColor(Color color)
        {
            // Pobiera wartości składowych RGB ze
            // struktury Color.
            string R = color.R.ToString();
            string G = color.G.ToString();
            string B = color.B.ToString();

            // Ustawia bieżący kolor w drugim
            // polu z rysunkiem.
            this.pictureBoxColor.BackColor = color;

            // Wyświetla wartości składowych RGB na 
            // etykietach.
            this.labelRValue.Text = R;
            this.labelGValue.Text = G;
            this.labelBValue.Text = B;
        }

        /// <summary>
        /// Generuje łańcuch znaków reprezentujący kod w języku C# lub Visual Basic 
        /// potrzebny do utworzenia egzemplarza struktury Color odpowiadającej przekazanej
        /// strukturze Color. Ten łańcuch znaków zostaje przypisany do pola _code
        /// kontrolki użytkownika.
        /// </summary>
        /// <param name="color">Kolor reprezentowany w kodzie.</param>
        /// <param name="isVB">Flaga logiczna określająca używany język:
        /// false oznacza język C#, a true — język Visual Basic</param>
        private void SetCode(Color color, bool isVB)
        {
            // Wczytuje ustawienia dodatku z rejestru.
            SetPropFromReg();

            string code = "";

            if (isVB)
            {
                code = "Dim color As Color = ";
            }
            else
            {
                code = "Color color = ";
            }

            code = code + "Color.FromArgb(" + color.R.ToString() + ", " +
                color.G.ToString() + ", " +
                color.B.ToString() + ");";

            _code = code;
            this.labelCode.Text = _code;

        }

        /// <summary>
        /// Wczytuje wartości z rejestru i w odpowiedni sposób ustawia 
        /// pole określające używany język.
        /// </summary>
        private void SetPropFromReg()
        {
            try
            {
                RegistryKey regKey =
                  Registry.CurrentUser.OpenSubKey(@"Software\Contoso\Addins\ColorSelector");
                string codeVal = (string)regKey.GetValue("Language", "CSharp");

                if (codeVal == "CSharp")
                {
                    _generateVB = false;
                }
                else
                {
                    _generateVB = true;
                }
            }
            catch (Exception ex)
            {
                // Błąd odczytue danych z rejestru. Ustawianie języka domyślnego (C#).
                _generateVB = false;
            }
        }
        #endregion

    }
}
