// Jan Bielecki - Visual 6.0 Threads
// Copyright 1999

#if !defined(_threads)
#define _threads
/*
Visual C++: Wymagane ustawienia dla bibliotek statycznych

Project / Settings // Link

1. Category: General

   Object/library modules:
       Zawarto tej klatki czasami znika. 
       Kompilator informuje "delete already defined".
       Klatka powinna wyszczeglnia
           kernel32.lib user32.lib gdi32.lib winspool.lib 
           comdlg32.lib advapi32.lib shell32.lib ole32.lib 
           oleaut32.lib uuid.lib odbc32.lib odbccp32.lib 
       Ponadto naley do niej doda
           libcmtd.lib   // w trybie Debug
           libcmt.lib    // w trybie Release

2. Category: Input

   Ignore libraries: 
           libcd.lib     // w trybie Debug
           libc.lib      // w trybie Release

Uwaga: Czasami libcmtd.lib albo libcmt.lib
       pojawia si w 2 miejscach. 
       Naley wwczas usun jeden egzemplarz.
*/

// ============================================= CONSOLE

#if defined(_CONSOLE)
#include <afxwin.h>

#define Infinite INFINITE

typedef HANDLE Handle;

#define Failed WAIT_FAILED         // -1
#define Signaled WAIT_OBJECT_0     // 0
#define Timeout WAIT_TIMEOUT       // 258
#define Abandoned WAIT_ABANDONED   // 128
#define Raised Signaled


// ============================================= Object

typedef Handle Object;

inline bool closeObject(Object &object)
{
    return CloseHandle(object) != 0;
}

// ============================================= (Object)


// ============================================= Process

typedef Handle Process;

inline Process getCurrentProcess(void)
{
    return GetCurrentProcess();
}

inline void exitProcess(int code =0)
{
    ExitProcess(code);
}

// ============================================= (Process)


// ============================================= Thread

typedef Handle Thread;

typedef int (*Run)(void *);
typedef LPTHREAD_START_ROUTINE Runner;

#define waitForThreads waitForObjects
#define closeThread closeObject

inline Thread createThread(Run pRun, void *pArg =0,
                           bool stop =false, int *pId = 0)
{
    int id = 0;
    Thread thread = 
        CreateThread(
            0,                           // no security
            0,                           // default stack
            (Runner)pRun,                // start function
            pArg,                        // start argument
            stop ? 1 : 0,                // start suspended
            (DWORD *)(pId ? pId : &id)   // thread ID
        );
    if(!thread)
        throw GetLastError();

    return thread;
}

inline void exitThread(int code =0)
{
    ExitThread(code);
}

inline int waitForThread(Thread &thread, int time =Infinite)
{
    return WaitForSingleObject(thread, time);
}

inline bool killThread(Thread &thread, int code =0)
{
    return TerminateThread(thread, code) != 0;
}

inline void sleep(int time =0)
{
    Sleep(time);
}

inline int suspendThread(Thread &thread)
{
    return SuspendThread(thread);
}

inline int resumeThread(Thread &thread)
{
    return resumeThread(thread);
}


inline Thread getCurrentThread(void)
{
    return GetCurrentThread();
}

// ============================================= (Thread)


// ============================================= Mutex

typedef HANDLE Mutex;

inline Mutex createMutex(bool signaled =true, 
                         char *pName =0)
{
    return CreateMutex(
               0,               // no security
               int(!signaled),  // initial ownership
               pName            // mutex name
           );
}

inline int waitForMutex(Mutex &mutex, int time =Infinite)
{
    return WaitForSingleObject(mutex, time);
}

inline bool freeMutex(Mutex &mutex)
{
    return ReleaseMutex(mutex) != 0;
}

// ============================================= (Mutex)


// ============================================= Section

typedef Mutex Section;

inline Section createSection(char *pName =0)
{
    return CreateMutex(
               0,        // no security
               false,    // initially owned
               pName     // mutex name
           );
}

inline void enterSection(Section &section, int time =Infinite, char *id ="")
{
    int reply = WaitForSingleObject(section, time);
}

inline void leaveSection(Section &section)
{
    ReleaseMutex(section);
}

inline void sleep(Section &section, int time =0)
{
    leaveSection(section);
    sleep(time);
    enterSection(section);
}

// ============================================= (Section)


// ============================================= Event

typedef HANDLE AnyEvent, AutoEvent, ManualEvent, Variable;

inline ManualEvent createManualEvent(bool signaled =false,
                                     char *pName =0)
{
    return CreateEvent(
               0,         // no security
               true,      // manual event
               signaled,  // initial state
               pName      // event name
           );
}

inline AutoEvent createAutoEvent(bool signaled =false, 
                                 char *pName =0)
{
    return CreateEvent(
               0,         // no security
               false,     // automatic event
               signaled,  // initial state
               pName      // event name
           );
}

inline int waitForEvent(AnyEvent &event, int time =Infinite)
{
    return WaitForSingleObject(event, time);
}

inline bool setEvent(AnyEvent &event)
{
    return SetEvent(event) != 0;
}

inline bool resetEvent(AnyEvent &event)
{
    return ResetEvent(event) != 0;
}

inline bool pulseEvent(AnyEvent &event)
{
    return PulseEvent(event) != 0;
}

inline int wait(AnyEvent &event, Section &section, 
                int time =Infinite)
{
    leaveSection(section);
    int reply = waitForEvent(event, time);
    resetEvent(event);
    enterSection(section);
    return reply;
}

inline bool notify(AnyEvent &event)
{
      return setEvent(event) != 0;
}

// ============================================= (Event)


// ============================================= Objects

inline int waitForObjects(
               int count, Variable *pVars,
               bool forAll =true,
               int time =Infinite
           )
{
    return WaitForMultipleObjects(count, pVars, 
                                  forAll, time);
}

// ============================================= (Objects)


// ====================================== Class extensions

typedef Handle   CThreads;
typedef AnyEvent CEvents;
#define waitForThreads waitForObjects
#define waitForEvents  waitForObjects

class CThread {
private:
    Thread thread;
public:
    CThread(Run pRun, void *pArg =0)         
    {
        thread = createThread(pRun, pArg);
    }
    operator Thread(void)
    {
        return thread;
    }
    int waitForThread(int time =Infinite)
    {
        return WaitForSingleObject(thread, time);
    }
    void exitThread(int code =0)
    {
        ::exitThread(code);
    }
};

class CSection {
private:
    Section section;
public:
    CSection(void)
    {
        section = createSection();
    }
    void enterSection(void)
    {
        ::enterSection(section);
    }
    void leaveSection(void)
    {
        ::leaveSection(section);
    }
};

class CAnyEvent {
protected:
    AnyEvent event;

    CAnyEvent(bool isManual, 
              bool signaled =false, char *pName =0)
    {
        if(isManual)
            event = createManualEvent(signaled, pName);
        else
            event = createAutoEvent(signaled, pName);
    }
public:
    operator AnyEvent(void)
    {
        return event;
    }
    bool setEvent(void)
    {
        return SetEvent(event) != 0;
    }
    bool resetEvent(void)
    {
        return ResetEvent(event) != 0;
    }
    bool pulseEvent(void)
    {
        return PulseEvent(event) != 0;
    }
    int wait(CSection &section, int time =Infinite)
    {
        section.leaveSection();
        int reply = WaitForSingleObject(event, time);
        resetEvent();
        section.enterSection();
        return reply;
    }
    int wait(int time =Infinite)
    {
        return WaitForSingleObject(event, time);
    }
    bool notify(void)
    {
        return SetEvent(event) != 0;
    }
};

class CManualEvent : public CAnyEvent {
public:
    CManualEvent(bool signaled =false, char *pName =0) 
        : CAnyEvent(true, signaled, pName)
    {
    }
};

class CAutoEvent : public CAnyEvent {
public:
    CAutoEvent(bool signaled =true, char *pName =0)
        : CAnyEvent(false, signaled, pName)
    {
    }
};

// ====================================== object extensions

#endif


// ========================================= WINDOWS

#if defined(_WINDOWS)

#include <afxwin.h>
#include <afxmt.h>

#define Infinite INFINITE

typedef HANDLE Handle;

#define Failed WAIT_FAILED         // -1
#define Signaled WAIT_OBJECT_0     // 0
#define Timeout WAIT_TIMEOUT       // 258
#define Abandoned WAIT_ABANDONED   // 128
#define Raised Signaled


typedef AFX_THREADPROC Worker;

#define Mutex HANDLE
typedef int (*Run)(void *);

#define Master(p) RUNTIME_CLASS(p)
typedef CRuntimeClass Runtime;

inline void sleep(int time =0)
{
    Sleep(time);
}

class CWorkerThread {
private:
    CWinThread *pThread;
public:
    CWorkerThread(Run pRun, void *pArg =0) 
        : pThread(AfxBeginThread((Worker)pRun, pArg))
    {
    }
    int waitForThread(int time =Infinite)
    {
        return WaitForSingleObject(pThread->m_hThread, time);
    }
    void exitThread(int code =0)
    {
        AfxEndThread(code);
    }
};

class CMasterThread {
private:
    CWinThread *pThread;
public:
    CMasterThread(Runtime *pRuntime) 
        : pThread(AfxBeginThread(pRuntime))
    {
    }
    int waitForThread(int time =Infinite)
    {
        return WaitForSingleObject(pThread->m_hThread, time);
    }
    void exitThread(int code =0)
    {
        AfxEndThread(code);
    }
};

class CSection : public CCriticalSection {
public:
    void enterSection(void)
    {
        Lock();
    }
    void leaveSection(void)
    {
        Unlock();
    }
};

class CAnyEvent : public CEvent {
protected:
    CAnyEvent(bool isManual, 
              bool signaled =false, char *pName =0)
        : CEvent(signaled, isManual, pName)
    {
    }
public:
    bool setEvent(void)
    {
        return SetEvent() != 0;
    }
    bool resetEvent(void)
    {
        return ResetEvent() != 0;
    }
    bool pulseEvent(void)
    {
        return PulseEvent() != 0;
    }
    int wait(CSection &section, int time =Infinite)
    {
        section.leaveSection();
        int reply = WaitForSingleObject(*this, time);
        resetEvent();
        section.enterSection();
        return reply;
    }
    int wait(int time =Infinite)
    {
        return WaitForSingleObject(*this, time);
    }
    bool notify(void)
    {
        return SetEvent() != 0;
    }
    operator CEvent *(void)
    {
        return (CEvent *)this;
    }
};

class CManualEvent : public CAnyEvent {
public:
    CManualEvent(bool signaled =false, char *pName =0) 
        : CAnyEvent(true, signaled, pName)
    {
    }
};

class CAutoEvent : public CAnyEvent {
public:
    CAutoEvent(bool signaled =true, char *pName =0)
        : CAnyEvent(false, signaled, pName)
    {
    }
};

typedef CEvent *CEvents;

class CMultiEvent {
private:
    CEvent **pEvents;
    int count;
public:
    CMultiEvent(CEvent **pEvents, int count)
        : pEvents(pEvents), count(count)
    {
    }
    int waitForEvents(bool forAll =true, int time =Infinite)
    {
        Handle *pHandles = new Handle [count];
        for(int i = 0; i < count ; i++) {
            pHandles[i] = pEvents[i]->m_hObject;
        }
        int reply = WaitForMultipleObjects(
                        count, pHandles, forAll, time
                    );
        delete [] pHandles;
        
        return reply;
    }
};

#endif

// ============================================= (WINDOWS)


// Common to CONSOLE & WINDOWS
// ============================================= Priority

typedef int Priority;
typedef HANDLE Process;
typedef HANDLE Thread;

#define RealTime REAL_TIME_PRIORITY_CLASS       // 256
#define High     HIGH_PRIORITY_CLASS            // 128
#define Regular  NORMAL_PRIORITY_CLASS          // 32
#define Idle     IDLE_PRIORITY_CLASS            // 64

#define Lowest   THREAD_PRIORITY_LOWEST         // -2
#define Lower    THREAD_PRIORITY_BELOW_NORMAL   // -1
#define Normal   THREAD_PRIORITY_NORMAL         //  0
#define Higher   THREAD_PRIORITY_ABOVE_NORMAL   // +1
#define Highest  THREAD_PRIORITY_HIGHEST        // +2


inline Priority getProcessPriority(Process &process)
{
    return GetPriorityClass(process);
}

inline bool setProcessPriority(Process &process, 
                               const Priority &priority)
{
    return SetPriorityClass(process, priority) != 0;
}

inline Priority getThreadPriority(Thread &thread)
{
    return GetThreadPriority(thread);
}

inline bool setThreadPriority(Thread &thread, 
                              const Priority &priority)
{
    return SetThreadPriority(thread, priority) != 0;
}

// ============================================= (Priority)

#endif
