Namespace commerce
    ' 
    ' Factory class for creating IQuantityobjects that express a weight.
    '
    Public Class WeightAmountFactory
        Inherits AbstractAmountFactory

        '
        ' The single instance of this class.
        '
        Private Shared myInstance As WeightAmountFactory _
            = New WeightAmountFactory()

        '
        ' A unit for measuring pounds.
        '
        Private ReadOnly POUND As MeasurementUnit

        '
        ' A unit for measuring ounces.
        '
        Private ReadOnly OUNCE As MeasurementUnit

        '
        ' A unit for measuring tons.
        '
        Private ReadOnly TON As MeasurementUnit

        '
        ' A unit for measuring Kilograms.
        '
        Private ReadOnly KG As MeasurementUnit

        '
        ' A unit for measuring grams.
        '
        Private ReadOnly GRAM As MeasurementUnit

        Private Sub New()
            MyBase.New("Weight", _
                        "This dimension for weights", _
                        "each", _
                        "Pounds", _
                        0, _
                        "#0.0 lbs.")
            POUND = Dimension.StandardUnit

            Dim ozDelegate As ConversionFactorProducer _
                = ConstantConverstionDelegate(1D / 16D)
            OUNCE = AddUnit("ounce", ozDelegate, -1, _
                            "ounces", "#0 oz.")

            Dim tonDelegate As ConversionFactorProducer _
                = ConstantConverstionDelegate(2000D)
            TON = AddUnit("ton", tonDelegate, -1, _
                            "tons", "#0.0## tons")

            Dim kgDelegate As ConversionFactorProducer _
                = ConstantConverstionDelegate(2.2046226D)
            KG = AddUnit("kg", kgDelegate, -1, _
                        "kilograms", "#0.0## kg")

            Dim gramDelegate As ConversionFactorProducer _
                = ConstantConverstionDelegate(0.0022046226D)
            GRAM = AddUnit("gram", gramDelegate, -1, _
                            "grams", "#0.0## g")
        End Sub

        '
        ' Return the single instance of this class.
        '
        Public Shared Function getInstance() As WeightAmountFactory
            Return myInstance
        End Function

        '
        ' Return an IQuantity that encapsulates the 
        ' given amount and unit of measurement.
        ' 
        ' Parameters:
        ' - theWeight
        '   The weight to be encapsulated.
        '
        ' - theUnit
        '   The unit of measurement to be encapsulated.
        '
        ' Return the requested IQuantity.
        '
        Public Function CreateWeight( _
                                ByVal theWeight As Decimal, _
                                ByVal theUnit As Weight) _
                        As IQuantity
            Dim unit As MeasurementUnit
            unit = WeightToMeasurementUnit(theUnit)
            Return CreateQuantity(theWeight, unit)
        End Function

        '
        ' Return a Quantity that encapsulates the given weight and unit of
        ' measurement.
        '
        ' Parameters:
        ' - theAmount
        '   The amount to be encapsulated.
        '
        ' - theWeight
        '   The unit of measurement to be encapsulated.
        '
        ' - theMaxPrecision
        '   The maximum number of decimal places to use for representing
        ' this quantity or -1 to indicate that there is no maximum.
        '
        ' Return the requested IQuantity.
        '
        Public Function CreateWeight( _
                            ByVal theAmount As Decimal, _
                            ByVal theWeight As Weight, _
                            ByVal theMaxPrecision As Integer) _
                        As IQuantity
            Dim unit As MeasurementUnit
            unit = WeightToMeasurementUnit(theWeight)
            Return CreateQuantity(theAmount, unit, _
                                    theMaxPrecision)
        End Function

        Private Function WeightToMeasurementUnit( _
                                ByVal theWeight As Weight) _
                        As MeasurementUnit
            Select Case (theWeight)
                Case Weight.Gram
                    Return GRAM

                Case Weight.Kg
                    Return KG

                Case Weight.Ounce
                    Return OUNCE

                Case Weight.Pound
                    Return POUND

                Case Weight.Ton
                    Return TON
            End Select
            Dim msg As String = "Unknown weight code"
            Throw New ApplicationException(msg)
        End Function

        '
        ' Convert the given Quantity of weight to the
        ' specified unit of measurement.
        ' 
        ' Parameters:
        ' - theQty
        '   The quantity to convert.
        '
        ' - theWeight
        '   The measurement unit to convert to.
        '
        ' - theMaxPrecision
        '   The maximum number of decimal places to use for representing 
        ' the converted quantity or -1 to indicate no maximum.
        '
        ' Returns the requested Quantity.
        '
        Public Function ConvertWeight( _
                            ByVal theQty As IQuantity, _
                            ByVal theWeight As Weight, _
                            ByVal theMaxPrecision As Integer) _
                        As IQuantity
            Dim unit As MeasurementUnit
            unit = WeightToMeasurementUnit(theWeight)
            Return MeasurementDimension.Convert(theQty, _
                                                unit, _
                                                theMaxPrecision)
        End Function

        '
        ' Convert the given Quantity of weight to the
        ' specified unit of measurement.
        ' 
        ' Parameters:
        ' - theQty
        '   The quantity to convert.
        '
        ' - theWeight
        '   The measurement unit to convert to.
        '
        ' Returns the requested Quantity.
        '
        Public Function ConvertWeight( _
                                ByVal theQty As IQuantity, _
                                ByVal theWeight As Weight) _
                        As IQuantity
            Dim unit As MeasurementUnit
            unit = WeightToMeasurementUnit(theWeight)
            Return MeasurementDimension.Convert(theQty, _
                                                unit, _
                                            unit.MaxPrecision)
        End Function
    End Class
End Namespace