Namespace commerce
    ' 
    ' This is the abstract base class for classes responsible for 
    ' creating quanities of a particular type of measurement.  Each 
    ' concrete subclass should correspond to a particular type of 
    ' measurment, such as money, eaches, linear measure...
    '
    ' For each concrete subclass, there will be a corresponding
    ' enumeration type whose members correspond to the different units
    ' of measurement that the concrete subclass supports.
    '
    ' Concrete subclasses will define a method with a name like
    ' CreateWeight or CreateMoney that takes at least one argument whose
    ' type is the enumeration
    '
    Public MustInherit Class AbstractAmountFactory
        Private ReadOnly myDimension As MeasurementDimension

        Protected Sub New(ByVal dimensionName As String, _
        ByVal dimensionDescription As String, _
        ByVal standardUnitName As String, _
        ByVal unitDescription As String, _
        ByVal unitPrecision As Integer, _
        ByVal unitFormatString As String)
            myDimension _
                = New MeasurementDimension(dimensionName, _
                                        dimensionDescription, _
                                        standardUnitName, _
                                        unitDescription, _
                                        unitPrecision, _
                                        unitFormatString)
        End Sub

        '
        ' The dimension quantities created by this object
        ' will measure.
        '
        Protected ReadOnly Property Dimension() As MeasurementDimension
            Get
                Return myDimension
            End Get
        End Property

        ' 
        ' A description of this object.
        ' 
        Public ReadOnly Property Description() As String
            Get
                Return myDimension.Description
            End Get
        End Property

        '
        ' Return the named MeasurementUnit.
        '
        Public Function LookupUnit(ByVal unitName As String) As MeasurementUnit
            Return Dimension.GetUnit(unitName)
        End Function

        '
        ' Add a unit to this dimension.
        '
        ' parameters:
        ' - theCannonicalName
        '   The unit's internal name. May be unsuitable for user interfaces.
        '
        ' - theConversionFactorProducer
        '   Multiplying the unit by its conversion factor will convert to
        '   this dimension's standard unit.  The unit's conversion factor
        '   is determined on the fly by calling this delegate.
        '
        ' - thePrecision
        '   The maximum number of digits of precision that should be used
        '   when doing arithmetic on quantities of this unit.
        '
        ' - theDescription
        '   The description of the unit.
        '
        ' - theFormatString
        '   This string is use to specify how a Quantity using this 
        '   MeasurementUnit will be formatted.
        ' 
        ' - This string may be any string that may be passed to the 
        '   Decimal.ToString method as a format string.
        '
        ' Returns the newly created MeasurementUnit object.
        '
        ' Exceptions:
        ' - System.Data.DuplicateNameException
        '   If this dimension already has a unit with the given name.
        '
        Protected Function AddUnit(ByVal theCannonicalName As String, _
        ByVal theConversionFactorProducer As ConversionFactorProducer, _
        ByVal thePrecision As Integer, _
        ByVal theDescription As String, _
        ByVal theFormatString As String) As MeasurementUnit
            Return Dimension.AddUnit(theCannonicalName, _
                theConversionFactorProducer, _
                thePrecision, _
                theDescription, _
                theFormatString)
        End Function

        '
        ' Return a delegate that can be called to return the given 
        ' conversion factor.
        ' 
        ' Parameters:
        ' - value
        '   A value to use as a constant conversion factor.
        ' 
        Protected Shared Function ConstantConverstionDelegate(ByVal value As Decimal) As ConversionFactorProducer
            Dim cnv As ConstantConversionFactor
            cnv = New ConstantConversionFactor(value)
            Return New ConversionFactorProducer(AddressOf cnv.GetValue)
        End Function

        '
        ' Create a Quantity object that encapsulates the given value and 
        ' unit of measurement.
        '
        ' The maximum number of decimal places to use for representing 
        ' this quanity is determined by the value of the given 
        ' MeasurementUnit object's MaxPrecision attribute at the time this
        ' constructor is called.
        ' 
        ' Parameters:
        ' - value
        '   The magnitude of the Quantity.
        '
        ' - theUnit
        '   The quantity's unit of measurment
        '
        ' Returns the newly created quantity.
        '
        Protected Function CreateQuantity(ByVal value As Decimal, _
        ByVal theUnit As MeasurementUnit) As IQuantity
            Return New DecimalQuantity(theUnit, value)
        End Function

        '
        ' Create a Quantity object that encapsulates the given value and 
        ' unit of measurement.
        '
        ' Parameters:
        ' - value
        '   The magnitude of the Quantity.
        '
        ' - theUnit
        '   The quantity's unit of measurment
        '
        ' - theMaxPrecision
        '   The maximum number of decimal places to use for representing 
        '   this quantity.
        '
        ' Returns the newly created quantity.
        '
        Protected Function CreateQuantity(ByVal value As Decimal, _
        ByVal theUnit As MeasurementUnit, _
        ByVal theMaxPrecision As Integer) As IQuantity
            Return New DecimalQuantity(theUnit, value, theMaxPrecision)
        End Function

        '
        ' Instances of this class encapsulate a constant conversion factor.
        '
        Private Class ConstantConversionFactor
            ' 
            ' The conversion factor this object encapsulates.
            ' 
            Private myValue As Decimal

            '
            ' Constructor
            '
            ' Parameters:
            ' - theValue
            '   The value encapsulated by this object.
            '
            Public Sub New(ByVal theValue As Decimal)
                myValue = theValue
            End Sub

            '
            ' Return the value encapsulated by this object.
            '
            ' Parameters:
            ' - args
            '   Ignored by this method.
            '
            Public Function GetValue(ByVal args As Object()) As Decimal
                Return myValue
            End Function
        End Class
    End Class
End Namespace

