Namespace commerce
  '
  ' Describe a unit of measurement.
  '
  Public Class MeasurementUnit
    '
    ' If the conversion factor for this unit has this value, then no 
    ' conversion is possible between this and other units.
    '
    Public Const NO_CONVERSION As Decimal = 0

    '
    ' Internal name of this unit. May not be suitable for
    ' user interfaces.
    '
    Private myCannonicalName As String

    '
    ' Delegate to the object that determines this unit's conversion 
    ' factor.  Multiply by this number to convert the the standard 
    ' unit of this unit's dimension.
    '
    Public myConversionFactorProducer As ConversionFactorProducer

    '
    ' The dimension that this unit is used to measure.
    '
    Private myDimension As MeasurementDimension

    '
    ' A description of this object.
    '
    Private myDescription As String

    '
    ' This string is use to specify how a Quantity using this 
    ' MeasurementUnit will be formatted.
    '
    ' This string may be any string that may be passed to the
    ' Decimal.ToString function as a format string.
    '
    Private myFormatString As String

    '
    ' The maximum number of digits of precision that should be used 
    ' when doing arithmetic on quantities of this unit.  If -1 no 
    ' artifical maximum is imposed.
    '
    Private myMaxPrecision As Integer

    '
    ' Constructor
    '
    ' Parameters:
    ' - theCannonicalName
    '   Internal name of this unit. May not be suitable for user 
    '   interfaces.
    '
    ' - theDimension
    '   The dimension that this unit is used to measure.
    ' 
    ' - theMaxPrecision
    '   The maximum number of digits of precision that should be used
    '   when doing arithmetic on quantities of this unit.  If -1 no
    '   artifical maximum is imposed.
    '
    ' - theConversionFactorProducer
    '   Multiply this unit by this value to convert to the
    '   MeasurementDimension object's standard unit.
    '
    ' - theDescription
    '   A description of this unit.
    '
    ' - theFormatString
    '   This string is used to specify how a Quantity using this
    '   MeasurementUnit will be formatted.
    ' 
    '   This string may be any string that may be passed to the 
        '   Decimal.ToString method as a format string.
    '
    Friend Sub New(ByVal theCannonicalName As String, _
                   ByVal theDimension As MeasurementDimension, _
                   ByVal theMaxPrecision As Integer, _
                   ByVal theConversionFactorProducer _
                     As ConversionFactorProducer, _
                   ByVal theDescription As String, _
                   ByVal theFormatString As String)
      myCannonicalName = theCannonicalName
      myDimension = theDimension
      myMaxPrecision = theMaxPrecision
      myConversionFactorProducer _
          = theConversionFactorProducer
      myDescription = theDescription
      myFormatString = theFormatString
    End Sub

    '
    ' Internal name of this unit. May not be suitable for
    ' user interfaces.
    '
    Public ReadOnly Property CannonicalName() As String
      Get
        Return myCannonicalName
      End Get
    End Property

    '
    ' A description of this object.
    '
    Public ReadOnly Property Description() As String
      Get
        Return myDescription
      End Get
    End Property

    '
    ' The dimension that this unit is used to measure.
    '
    Public ReadOnly Property Dimension() As MeasurementDimension
      Get
        Return myDimension
      End Get
    End Property

    '
    ' This string is use to specify how a Quantity using this 
    ' MeasurementUnit will be formatted.
    '
    ' This string may be any string that may be passed to the 
        ' Decimal.ToString method as a format string.
    '
    Public ReadOnly Property FormatString() As String
      Get
        Return myFormatString
      End Get
    End Property

    '
    ' Multiply by the number returned by this method to convert this 
    ' unit to the standard unit of this unit's dimension.
    '
    ' The parameters for the conversion are supplied in an object 
    ' array.  The parameters to these delegates can vary.  However, 
    ' they may should be the same for all of the units that belong 
    ' to the same dimension.
    ' 
    Friend Function GetConversionFactor(ByVal args() As Object) As Decimal
      Return myConversionFactorProducer(args)
    End Function

    ' 
    ' Return true if this unit equals the given object.
    '
    ' Parameters:
    ' - obj
    '   The object to compare to this object.
    '
    ' Returns true if the given object is a MeasurementUnit that has 
    ' the same name and dimension as this object.
    '
    Public Overloads Overrides Function Equals(ByVal obj As Object) As Boolean
      If (Me Is obj) Then
        Return True
      End If
      If (TypeOf obj Is MeasurementUnit) Then
        Dim that As MeasurementUnit = DirectCast(obj, MeasurementUnit)
        If (Dimension.Equals(that.Dimension) _
            And CannonicalName.Equals(that.CannonicalName)) Then
          Return True
        End If
      End If
      Return False
    End Function

    '
    ' Return a hash code for this object.
    '
    Public Overrides Function GetHashCode() As Integer
      Return CannonicalName.GetHashCode() _
         Xor Dimension.GetHashCode()
    End Function

    '
    ' Return a string reprsentation of this object.
    '
    Public Overrides Function ToString() As String
            Return "unit[name=" & CannonicalName _
          & "; dimension=" & Dimension.ToString() & "]"
    End Function

    '
    ' The maximum number of digits of precision that should be used 
    ' when doing arithmetic on quantities of this unit.  If -1 no 
    ' artifical maximum is imposed.
    '
    Public ReadOnly Property MaxPrecision() As Integer
      Get
        Return 0
      End Get
    End Property
  End Class
End Namespace
