Imports System.Collections
Imports System.Data

Namespace commerce
    '
    ' Model a kind of measurement such as linear, volume, time, ...
    '
    ' A MeasurementDimension object has a MeasurementUnit object 
    ' associated with it that is its standard unit.  The standard unit
    ' is the unit that is the basis conversion between different units
    ' that are associated with this MeasurementDimension.
    ' 
    ' The conversion facator associated with each unit is the number you
    ' multiply by to convert that unit to the standard unit.  The
    ' conversion factor associated with the standard unit is always 1.0.
    ' 
    Public Class MeasurementDimension
        Implements IDescription

        '
        ' Internal name of this dimension. May not be
        ' suitable for user interfaces.
        '
        Private myCannonicalName As String

        '
        ' A description of this object.
        '
        Private myDescription As String

        Private Shared ReadOnly identityConversionFactorProducer _
                                As ConversionFactorProducer _
            = New ConversionFactorProducer(AddressOf IdentityConversionFunction)

        '
        ' A MeasurementDimension object has a MeasurementUnit object 
        ' associated with it that is its standard unit.  The standard unit
        ' is the unit that is the basis conversion between different units
        ' that are associated with this MeasurementDimension.  The 
        ' conversion facator associated with each unit is the number you 
        ' multiply by to convert that unit to the standard unit.  The
        ' conversion factor associated with the standard unitis always 1.0.
        '
        Private myStandardUnit As MeasurementUnit

        '
        ' A dictionary of the units for this dimension keyed by name.
        '
        Private unitDictionary As Hashtable = New Hashtable()

        '
        ' Constructor
        '
        ' Parameters
        ' - dimensionName
        '   Internal name of this dimension. May not be suitable for user 
        '   interfaces.
        '
        ' - dimensionDescription
        '   A description of this dimension.
        '
        ' - standardUnitName
        '   Internal name of this dimension's standard unit.
        '
        ' - unitDescription
        '   A description of this dimension's standard unit.
        '
        ' - unitPrecision
        '   The maximum number of digits of precision that should be used
        '   when doing arithmetic on quantities of this unit.
        '
        ' - unitFormatString
        '   This string is use to specify how a Quantity using this
        '   MeasurementUnit will be formatted.
        ' 
        '   This string may be any string that may be passed to the 
        '   Decimal.ToString method as a format string.
        '
        Public Sub New(ByVal dimensionName As String, _
                    ByVal dimensionDescription As String, _
                    ByVal standardUnitName As String, _
                    ByVal unitDescription As String, _
                    ByVal unitPrecision As Integer, _
                    ByVal unitFormatString As String)
            myCannonicalName = dimensionName
            myDescription = unitDescription
            myStandardUnit _
                = New MeasurementUnit(standardUnitName, Me, _
                                        unitPrecision, _
                                        identityConversionFactorProducer, _
                                        unitDescription, _
                                        unitFormatString)
            unitDictionary.Add(standardUnitName, _
                                myStandardUnit)
        End Sub

        '
        ' Internal name of this dimension. May not be
        ' suitable for user interfaces.
        '
        Public ReadOnly Property CannonicalName() As String
            Get
                Return myCannonicalName
            End Get
        End Property

        '
        ' A description of this object.
        '
        Public ReadOnly Property Description() As String _
                Implements IDescription.Description
            Get
                Return myDescription
            End Get
        End Property

        '
        ' A MeasurementDimension object has a MeasurementUnit object
        ' associated with it that is its standard unit.  The standard unit
        ' is the unit that is the basis conversion between different units
        ' associated with this MeasurementDimension.  The conversion factor
        ' associated with each unit is the number you multiply by to 
        ' convert that unit to the standard unit.  The conversion factor 
        ' associated with the standard unit is always 1.0.
        '
        Public ReadOnly Property StandardUnit() As MeasurementUnit
            Get
                Return myStandardUnit
            End Get
        End Property

        '
        ' Add a unit to this dimension.
        '
        ' Parameters:
        ' - theCannonicalName
        '   An internal name of the unit. May not be suitable for user 
        '   interfaces.
        '
        ' - theConversionFactorProducer
        '   Multiplying the unit by its conversion factor will convert to 
        '   this dimension's standard unit.  The unit's conversion factor
        '   is determined on the fly by calling this delegate.
        '
        ' - thePrecision
        '   The maximum number of digits of precision that should be used
        '   when doing arithmetic on quantities of this unit.
        '
        ' - theDescription
        '   The description of the unit.
        '
        ' - theFormatString
        '   This string is used to specify how a Quantity with this 
        '   MeasurementUnit will be formatted.
        '
        '   This string may be any string that may be passed to the 
        '   Decimal.ToString method as a format  string.
        '
        ' Exceptions:
        ' - System.Data.DuplicateNameException
        '   If this dimension already has a unit with the given name.
        '
        ' Returns the newly created MeasurementUnit object.
        '
        Public Function AddUnit(ByVal theCannonicalName As String, _
                                ByVal theConversionFactorProducer _
                                As ConversionFactorProducer, _
                                ByVal thePrecision As Integer, _
                                ByVal theDescription As String, _
                                ByVal theFormatString As String) As MeasurementUnit
            If (unitDictionary.Contains(theCannonicalName)) Then
                Dim msg As String = theCannonicalName
                Throw New DuplicateNameException(msg)
            End If
            Dim thisUnit As MeasurementUnit
            thisUnit = New MeasurementUnit(theCannonicalName, _
                                            Me, _
                                            thePrecision, _
                                            theConversionFactorProducer, _
                                            theDescription, _
                                            theFormatString)
            unitDictionary.Add(theCannonicalName, thisUnit)
            Return thisUnit
        End Function

        '
        ' Return this dimension's unit that has the given name or Nothing
        ' if there is no unit with the given name.
        '
        ' Parameters:
        ' - theCanonnicalName
        '   The internal name of the unit.
        '
        Public Function GetUnit(ByVal theCanonnicalName As String) _
                    As MeasurementUnit
            Dim unitObject As Object
            unitObject = unitDictionary(theCanonnicalName)
            Return DirectCast(unitObject, MeasurementUnit)
        End Function

        '
        ' Return a string representation of this measurement dimension.
        '
        Public Overrides Function ToString() As String
            Return "dimension[name=" & myCannonicalName _
                & "; standard unit=" _
                & myStandardUnit.CannonicalName & "]"
        End Function

        '
        ' Convert the given quantity to a quantity measured
        ' in the given measurement unit.
        '
        ' Parameters:
        ' - qty
        '   The quantity to convert to the given unit of measurement.
        '
        ' - toUnit
        '   The unit of measurement to convert the given quantity to.
        '
        ' - theMaxPrecision
        '   The maximum number of digits of after the decimal place to use
        '   to requresent the converted quantity.
        ' 
        ' - cnvParams
        '   An array of the parameters needed for the specific dimension.
        '
        ' Returns a Quantity measured in the given unit of measurement.
        '
        ' Exception:
        ' - ConversionException
        '   If the requested conversion cannot be performed.
        '
        Public Shared Function _
            Convert(ByVal qty As IQuantity, _
                    ByVal toUnit As MeasurementUnit, _
                    ByVal theMaxPrecision As Integer, _
                    ByVal ParamArray cnvParams() As Object) As IQuantity
            If (qty.Unit.Equals(toUnit)) Then
                ' Same unit, so no converstion needed.
                Return qty
            End If
            Dim thisDimension As MeasurementDimension
            thisDimension = toUnit.Dimension
            If (Not qty.Unit.Dimension.Equals(thisDimension)) Then
                Dim msg As String
                msg = "Cannot convert from a " _
                    & qty.Unit.Dimension.CannonicalName _
                    & " to a " & thisDimension.CannonicalName
                Throw New ConversionException(msg)
            End If
            Dim fromFactor As Decimal
            fromFactor = qty.Unit.GetConversionFactor(cnvParams)
            Dim toFactor As Decimal
            toFactor = toUnit.GetConversionFactor(cnvParams)
            If (fromFactor = MeasurementUnit.NO_CONVERSION) Then
                Inconvertible(qty.Unit)
            End If
            If (toFactor = MeasurementUnit.NO_CONVERSION) Then
                Inconvertible(toUnit)
            End If

            Dim amount As Decimal
            amount = qty.Amount * fromFactor / toFactor
            Return New DecimalQuantity(toUnit, amount, theMaxPrecision)
        End Function

        '
        ' Throw a ConversionException
        '
        ' Parameters:
        ' - unit
        '   The unit that should be referenced in the message.
        '
        Private Shared Sub Inconvertible(ByVal unit As MeasurementUnit)
            Dim msg As String
            msg = "Cannot convert to or from " _
                & unit.CannonicalName
            Throw New ConversionException(msg)
        End Sub

        '
        ' This function is called through a delegate and always returns 
        ' one.  It is intended to provide the conversion factor for 
        ' standard units.
        ' 
        ' Parameters:
        ' - args
        '   An array of arguments that the caller thinks are needed for the
        '   conversion can be passed into this parameter.
        '
        Private Shared Function _
                IdentityConversionFunction(ByVal args() As Object) As Decimal
            Return 1D
        End Function
    End Class
End Namespace
