Imports System.Collections

Namespace DataStructure
    '
    ' This class is a doubly linked list data structure.  It is not
    ' intended to be a general-purpose data structure.  It is intended 
    ' to by used to maintain an ordering of objects that are part of 
    ' another data structure.
    '
    Public Class DoubleLinkedList

        '
        ' The first node in the list.
        '
        Private myFirst As IDoubleLink

        '
        ' The last node in the list.
        '
        Private myLast As IDoubleLink

        '
        ' The number of nodes in this doubly linked list.
        '
        Private myCount As Integer

        Private myModificationCount As Long

        '
        ' Constructor
        '
        Public Sub New()
        End Sub

        '
        ' The first member of this doubly linked list.
        '
        Public ReadOnly Property First() As IDoubleLink
            Get
                Return myFirst
            End Get
        End Property

        '
        ' The last member of this doubly linked list.
        '
        Public ReadOnly Property Last() As IDoubleLink
            Get
                Return myLast
            End Get
        End Property

        '
        ' The number of nodes in this doubly linked list.
        '
        Public ReadOnly Property Count() As Integer
            Get
                Return myCount
            End Get
        End Property

        '
        ' This is always false to indicate that access to
        ' this object is not thread-safe.
        '
        Public ReadOnly Property IsSynchronized() As Boolean
            Get
                Return False
            End Get
        End Property

        '
        ' Return the object that can be used to synchronize
        ' access to this object.
        '
        Public ReadOnly Property SyncRoot() As Object
            Get
                Return Me
            End Get
        End Property

        '
        ' Add the given object to the beginning of the linked list.
        ' 
        ' Parameters:
        ' - node
        '   The object to be added to the beginning of the list.
        '
        Public Sub AddFirst(ByVal node As IDoubleLink)
            If (myFirst Is Nothing) Then
                AddToEmpty(node)
            Else
                node.Nxt = myFirst
                myFirst.Previous = node
                node.Previous = Nothing
                myFirst = node
                myCount += 1
                myModificationCount += 1
            End If
        End Sub

        '
        ' Add the given object to the end of the linked list.
        '
        ' Parameters:
        ' - node
        '   The object to be added to the end of the list.
        '
        Public Sub AddLast(ByVal node As IDoubleLink)
            If (myLast Is Nothing) Then
                AddToEmpty(node)
            Else
                node.Previous = myLast
                myLast.Nxt = node
                node.Nxt = Nothing
                myLast = node
                myCount += 1
                myModificationCount += 1
            End If
        End Sub

        '
        ' Insert the new node before the reference node.
        '
        ' Parameters:
        ' - referenceNode
        '   The node to insert before.
        '
        ' - newNode
        '   The node to insert.
        '
        Public Sub InsertBefore(ByVal referenceNode As IDoubleLink, _
                                ByVal newNode As IDoubleLink)
            If (referenceNode Is myFirst) Then
                AddFirst(newNode)
            Else
                Dim refPrev As IDoubleLink = referenceNode.Previous
                newNode.Nxt = referenceNode
                newNode.Previous = refPrev
                referenceNode.Previous = newNode
                refPrev.Nxt = newNode
                myCount += 1
                myModificationCount += 1
            End If
        End Sub

        '
        ' Insert the new node after the reference node.
        '
        ' Parameters:
        ' - referenceNode
        '   The node to insert after.
        '
        ' - newNode
        '   The node to insert.
        '
        Public Sub insertAfter(ByVal referenceNode As IDoubleLink, _
                            ByVal newNode As IDoubleLink)
            If (referenceNode Is myLast) Then
                AddLast(newNode)
            Else
                Dim refNext As IDoubleLink = referenceNode.Nxt
                newNode.Previous = referenceNode
                newNode.Nxt = refNext
                referenceNode.Nxt = newNode
                refNext.Previous = newNode
                myCount += 1
                myModificationCount += 1
            End If
        End Sub

        '
        ' Remove the given node from the list.  This method does not check
        ' to see if the node is actually in the list.  If the node is not
        ' in the list, the list may become corrupted.
        '
        ' Parameters:
        ' - node
        '   The node to remove from this doubly linked list.
        '
        Public Sub remove(ByVal node As IDoubleLink)
            If (node Is myFirst) Then
                myFirst = node.Nxt
            Else
                node.Previous.Nxt = node.Nxt
            End If
            If (node Is myLast) Then
                myLast = node.Previous
            Else
                node.Nxt.Previous = node.Previous
            End If
            myCount -= 1
            node.Previous = Nothing
            node.Nxt = Nothing
            myModificationCount -= 1
        End Sub

        '
        ' Copy the nodes of this doubly linked list to the given array.
        '
        ' Parameters:
        ' - theArray
        '   The array to copy the nodes to.
        ' - index
        '   The copying operation should start at this index of the array.
        '
        Public Sub CopyTo(ByVal theArray As Array, _
                        ByVal index As Integer)
            If (theArray Is Nothing) Then
                Throw New ArgumentNullException("array")
            End If
            If (index < 0) Then
                Dim indexValue As String = index.ToString()
                Throw New ArgumentOutOfRangeException("index", _
                                                    indexValue)
            End If
            If (theArray.Rank > 1) Then
                Dim msg As String = "Array is multidimensional"
                Throw New ArgumentException(msg, "array")
            End If
            If (index >= theArray.Length) Then
                Dim msg As String = "index >= length of the array."
                Throw New ArgumentException(msg, "index")
            End If
            If (Count > theArray.Length - index) Then
                Dim msg As String = "Not enough room to copy!"
                Throw New ArgumentException(msg)
            End If
            Dim thisNode As IDoubleLink = First
            While thisNode IsNot Nothing
                theArray.SetValue(thisNode, index)
                thisNode = thisNode.Nxt
            End While
        End Sub

        '
        ' Make this double linked list empty.
        '
        Public Sub Clear()
            myFirst = Nothing
            myLast = Nothing
            myCount = 0
            myModificationCount += 1
        End Sub

        '
        ' Assume the linked list is empty and add the given node.
        '
        ' Parameters:
        ' - node
        '   The node to add to this doubly linked list.
        '
        Private Sub AddToEmpty(ByVal node As IDoubleLink)
            node.Previous = Nothing
            node.Nxt = Nothing
            myFirst = node
            myLast = node
            myModificationCount += 1
        End Sub

        '
        ' Return an IEnumerator object to navigate this data structure.
        Public Function GetEnumerator() As IEnumerator
            Dim theDelegate As ModificationCountDelegate _
                = New ModificationCountDelegate(AddressOf GetModCount)

            '@@@From TE2: DoubleLinkedListEnumerator does not implement
            'IEnumerator. It inherits from AbstractEnumerator, which has
            'comments that say it helps implement IEnumerator but it doesn't
            'do so either. @@@
            Return Nothing

            'Return New DoubleLinkedListEnumerator(Me, _
            '                                        theDelegate)
        End Function

        Private Function GetModCount() As Long
            Return myModificationCount
        End Function

        Private Class DoubleLinkedListEnumerator
            Inherits AbstractEnumerator

            ' 
            ' The DoubleLinkedList this object will navigate.
            '
            Private myList As DoubleLinkedList

            '
            ' The next node that will become the current node.
            '
            Private myNext As IDoubleLink

            '
            ' Constructor
            '
            ' Parameters:
            ' - theList
            '   The DoubleLinkedList this object will navigate.
            '
            ' - getModificationCount
            '   The delegate for getting the modification count of the 
            '   underlying object this object is responsible for navigating.
            Friend Sub New(ByVal theList As DoubleLinkedList, _
                            ByVal getModificationCount As ModificationCountDelegate)
                MyBase.new(getModificationCount)
                myList = theList
                ResetImpl()
            End Sub

            ' 
            ' Point this object at the beginning of the list
            ' it will be navigating.
            ' 
            Protected Overrides Sub ResetImpl()
                myNext = myList.First
            End Sub

            Protected Overrides Function NextElement(ByRef node As Object) _
                                        As Boolean
                If (myNext Is Nothing) Then
                    Return False
                End If
                node = myNext
                myNext = myNext.Nxt
                Return True
            End Function
        End Class
    End Class
End Namespace

