Namespace commerce
    Public Structure DecimalQuantity
        Implements IQuantity

        '
        ' The amount that this quantity represents.
        '
        Private myAmount As Decimal

        '
        ' The measurement unit of this quantity.
        '
        Private myUnit As MeasurementUnit

        '
        ' The amount of the specified measurement unit this quantity
        ' represents.
        '
        Public ReadOnly Property Amount() As Decimal _
                    Implements IQuantity.Amount
            Get
                Return myAmount
            End Get
        End Property

        '
        ' The unit of measurment for this quantity.
        '
        Public ReadOnly Property Unit() As MeasurementUnit _
            Implements IQuantity.Unit
            Get
                Return myUnit
            End Get
        End Property

        '
        ' The number of decimal places of precision represented by
        ' this quantity.  Setting this to a smaller value will cause
        ' any excess precission to be rounded off.
        '
        Public Property MaxPrecision() As Integer
            Get
                Return myMaxPrecision
            End Get
            Set(ByVal Value As Integer)
                myMaxPrecision = Value
                Round()
            End Set
        End Property

        '
        ' The maximum number of decimal places that will be used
        ' to represent this quantity.
        '
        Private myMaxPrecision As Integer

        '
        ' This constructor uses the precision provided by the given 
        ' MeasurementUnit to determine the number of decimal places to 
        ' used for representing this quantity.
        '
        ' Parameters:
        ' - theUnit
        '   The unit this quantity is measured in.
        '
        ' - theAmount
        '   The amount of the specified unit of measurement this
        '   quantity represents.
        '
        Public Sub New(ByVal theUnit As MeasurementUnit, _
                    ByVal theAmount As Decimal)
            Me.New(theUnit, theAmount, theUnit.MaxPrecision)
        End Sub

        '
        ' This constructor is passed an explicit number of
        ' decimal places to use for representing this
        ' quantity.
        '
        ' Parameters:
        ' - theUnit
        '   The unit this quantity is measured in.
        '
        ' - theAmount
        '   The amount of the specified unit of measurement this
        '   quantity represents.
        '
        ' - theMaxPrecision
        '   The maximum number of decimal places to use for
        '   representing this quantity.
        Public Sub New(ByVal theUnit As MeasurementUnit, _
                    ByVal theAmount As Decimal, _
                    ByVal theMaxPrecision As Integer)
            myUnit = theUnit
            myAmount = theAmount
            myMaxPrecision = theMaxPrecision
            Round()
        End Sub


        ' 
        ' Return a quantity that is the sum of this Quantity and the given
        ' Quantity;
        ' 
        ' Parameters:
        ' - that
        '   The quantity to add to this one.
        '
        ' Returns the sum of the two quantities
        '
        ' Exceptions:
        ' - ConversionException
        '   If this quantity and the given quantity
        '   have different units of measurement.
        '
        Public Function add(ByVal that As IQuantity) _
                        As IQuantity Implements IQuantity.add
            checkUnit(that)
            Dim sum As Decimal = Amount + that.Amount
            Return New DecimalQuantity(Unit, sum, MaxPrecision)
        End Function

        '
        ' Return a Quantity that is the difference between this Quantity 
        ' and the given Quantity.
        ' 
        ' Parameters:
        ' - that
        '   The Quantity to subtactfrom this one.
        '
        ' Exceptions
        ' - ConversionException
        '   If this quantity and the given quantity have different units
        '   of measurement.
        '
        Public Function subtract(ByVal that As IQuantity) _
                    As IQuantity Implements IQuantity.subtract
            checkUnit(that)
            Dim difference As Decimal = Amount - that.Amount
            Return New DecimalQuantity(Unit, difference, MaxPrecision)
        End Function

        '
        ' Compare this quantity with the given quantity.
        '
        ' Parameters:
        ' - obj
        '   A quantity to compare to this one.
        '
        ' The return value has these meanings:
        '     Value              Meaning
        '     Less than zero     This quantity is less than the given 
        '                        quantity.
        '     Zero               This quantity is equal to the given 
        '                        quantity.
        '     Greater than zero  This quantity is greater than the given 
        '                        quantity.
        '
        ' Exceptions:
        ' - ConversionException
        '   If this quantity and the quantity amount have different units
        '   of measurement.
        '
        ' - ArgumentException
        '   If that is not a Quantity.
        '
        Public Function CompareTo(ByVal obj As Object) _
                        As Integer Implements IQuantity.CompareTo
            If (TypeOf obj Is IQuantity) Then
                Dim that As DecimalQuantity
                that = DirectCast(obj, DecimalQuantity)
                checkUnit(that)
                Return Amount.CompareTo(that.Amount)
            Else
                Dim msg As String = obj.GetType().FullName
                Throw New ArgumentException(msg)
            End If
        End Function

        '
        ' Throw a ConversionException\ if the given Quantity object has 
        ' different units of measurement than this one.
        '
        ' Parameters:
        ' - that
        '   The Quantity object that must have the same measurement unit
        '   as this one.
        '
        Private Sub checkUnit(ByVal that As IQuantity)
            If (Not Me.Unit.Equals(that.Unit)) Then
                Dim msg As String = "Different unit of measurement: "
                Throw New ConversionException(msg)
            End If
        End Sub

        '
        ' Multiply this quantity by the given value.
        '
        ' Parameters:
        ' - x
        '   The multiplier.
        ' 
        ' Returns the product.
        '
        Public Function multiply(ByVal x As Integer) _
                    As IQuantity Implements IQuantity.multiply
            Dim product As Decimal = Amount * x
            Return New DecimalQuantity(Unit, product, MaxPrecision)
        End Function

        '
        ' Multiply this quantity by the given value.
        '
        ' Parameters:
        ' - x
        '   The multiplier.
        ' 
        ' Returns the product.
        '
        Public Function multiply(ByVal x As Decimal) _
                    As IQuantity Implements IQuantity.multiply
            Dim product As Decimal = Amount * x
            Return New DecimalQuantity(Unit, product, MaxPrecision)
        End Function

        '
        ' Multiply this quantity by the given value.
        '
        ' Parameters:
        ' - x
        '   The multiplier.
        ' 
        ' Returns the product.
        '
        Public Function multiply(ByVal x As Double) _
                    As IQuantity Implements IQuantity.multiply
            Dim product As Decimal = Amount * CDec(x)
            Return New DecimalQuantity(Unit, product, MaxPrecision)
        End Function

        '
        ' Round this Quantity to its maximim precision.
        '
        Private Sub Round()
            ' if max precision less than zero, then there is
            ' no maximum precision.
            If (myMaxPrecision > 0) Then
                myAmount = Decimal.Round(myAmount, myMaxPrecision)
            End If
        End Sub

        '
        ' Divide this quantity by the given value.
        '
        ' Parameters:
        ' - x
        '   The divisor.
        '
        ' Returns the quotient.
        '
        Public Function divide(ByVal x As Integer) _
                    As IQuantity Implements IQuantity.divide
            Dim quotient As Decimal = Amount / x
            Return New DecimalQuantity(Unit, quotient, MaxPrecision)
        End Function

        '
        ' Divide this quantity by the given value.
        '
        ' Parameters:
        ' - x
        '   The divisor.
        '
        ' Returns the quotient.
        '
        Public Function divide(ByVal x As Double) _
                    As IQuantity Implements IQuantity.divide
            Dim quotient As Decimal = Amount / CDec(x)
            Return New DecimalQuantity(Unit, quotient, MaxPrecision)
        End Function

        '
        ' Divide this quantity by the given value.
        '
        ' Parameters:
        ' - x
        '   The divisor.
        '
        ' Returns the quotient.
        '
        Public Function divide(ByVal x As Decimal) _
                    As IQuantity Implements IQuantity.divide
            Dim quotient As Decimal = Amount / x
            Return New DecimalQuantity(Unit, quotient, MaxPrecision)
        End Function

        ' 
        ' Return true if the given object is a Quantity and it has the same
        ' amount and unit of measurement as this amount.  Any differences
        ' in the value of the MaxPrecision property are ignored.
        '
        ' Parameters:
        ' - obj
        '   The object to compare this one to.
        '
        Public Overloads Overrides Function Equals(ByVal obj As Object) As Boolean
            If (TypeOf obj Is DecimalQuantity) Then
                Dim that As DecimalQuantity = DirectCast(obj, DecimalQuantity)
                Return myAmount = that.myAmount _
                    AndAlso myUnit.Equals(that.myUnit)
            End If
            Return False
        End Function

        '
        ' Return a hashcode for this object
        '
        Public Overrides Function GetHashCode() As Integer
            Return myAmount.GetHashCode() _
                Xor myUnit.GetHashCode()
        End Function

        ' 
        ' Returns true if the amount in this quantity is zero.
        ' 
        Public Function isZero() As Boolean _
                Implements IQuantity.isZero
            Return myAmount = 0
        End Function

        '
        ' Return a string representation of this quantity.
        '
        Public Overrides Function ToString() As String
            Return Amount.ToString(Unit.FormatString)
        End Function
    End Structure
End Namespace

