Namespace DataStructure
    '
    ' Abstract class to help implement the IEnumerator interface.
    '
    Public MustInherit Class AbstractEnumerator
        '
        ' Delegate this object will use to get the modification count of
        ' the data structure it is responsible for navigating.  Every time
        ' the data structure is modified, this count is supposed to be 
        ' incremented.
        '
        Private getModificationCount As ModificationCountDelegate

        '
        ' The modification count of the underlying data
        ' structure at the time this object was created.
        '
        Private originalCount As Long

        '
        ' The current member of the data structure being navigated.
        '
        Private myCurrent As Object = Nothing

        '
        ' If this is false, the current property is not mapped to any 
        ' member of the underlying data structure.
        '
        Private currentIsMapped As Boolean = False

        '
        ' Constructor
        '
        Public Sub New(ByVal theDelegate As ModificationCountDelegate)
            getModificationCount = theDelegate
            originalCount = getModificationCount()
        End Sub

        '
        ' The current member of the underlying data structure.
        '
        ' Exceptions:
        ' - System.InvalidOperationException
        '   Thrown if the current position in the data structure does not
        '   correspond to any element.  This is true when this object is 
        '   first created, after its MoveNext function returns false and 
        '   after its Reset subroutine is called.
        '
        Public ReadOnly Property Current() As Object
            Get
                If (Not currentIsMapped) Then
                    Dim msg As String = "No current data structure member."
                    Throw New InvalidOperationException(msg)
                End If
                Return myCurrent
            End Get
        End Property

        '
        ' If this is true, the underlying data structure has been modified.
        '
        Public ReadOnly Property InvalidFlag() As Boolean
            Get
                Return getModificationCount() <> originalCount
            End Get
        End Property

        '
        ' Throw an exception if the underlying
        ' data structure has been modified.
        '
        Private Sub CheckModified()
            If (InvalidFlag) Then
                Dim msg As String
                msg = "Underlying data structure modified"
                Throw New InvalidOperationException(msg)
            End If
        End Sub

        '
        ' This method is responsible for navigating to the
        ' next member of the underlying data structure.
        '
        ' Parameters
        ' - obj
        '   The variable to update with a reference to the new current
        '   data structure member.
        '
        ' Returns true if navigation to the next data structure member was
        ' successful or false if we are past the last member of the data 
        ' structure.
        '
        Protected MustOverride _
                Function NextElement(ByRef obj As Object) As Boolean

        '
        ' Move this enumerator to the next member of the
        ' underlying data structure.
        '
        ' Returns true if this enumerator advanced to the next data 
        ' structure element or false if this enumerator is past the last 
        ' member of the data structure.
        '
        ' Exceptions:
        ' - System.InvalidOperationException
        '   Thrown if the underlying data structure has been modified.
        Public Function MoveNext() As Boolean
            CheckModified()
            currentIsMapped = NextElement(myCurrent)
            Return currentIsMapped
        End Function

        '
        ' Navigate to before the first member of the 
        ' underlying data structure.
        '
        Protected MustOverride Sub ResetImpl()

        '
        ' Navigate to before the first member of the 
        ' underlying data structure.
        '
        '
        ' Exceptions:
        ' - System.InvalidOperationException
        '   Thrown if the underlying data structure has been modified.
        '
        Public Sub Reset()
            CheckModified()
            ResetImpl()
        End Sub
    End Class
End Namespace

