Namespace commerce
    '
    ' Factory class for creating IQuantity objects that
    ' express a quantity of discrete real world objects.
    '
    Public Class UnitAmountFactory
        Inherits AbstractAmountFactory

        '
        ' The single instance of this class.
        '
        Private Shared ReadOnly myInstance As UnitAmountFactory _
        = New UnitAmountFactory()

        '
        ' A unit for measuring individual items.
        '
        Private ReadOnly EACHES As MeasurementUnit

        '
        ' A unit for measuring individual items in groups of twelve.
        '
        Private ReadOnly DOZEN As MeasurementUnit

        '
        ' A unit for measuring individual items in groups of 144.
        '
        Private ReadOnly GROSS As MeasurementUnit

        '
        ' Private constructor to ensure that only this class's GetInstance
        ' function can instantiate this class.
        '
        Private Sub New()
            MyBase.New("Eaches", _
                    "Dimension for counting discrete objects", _
                        "Each", _
                        "individual real discrete objects", _
                        0, _
                        "#0;(#0)")
            EACHES = Dimension.StandardUnit

            Dim dozenDelegate As ConversionFactorProducer _
                = ConstantConverstionDelegate(1D / 12D)
            DOZEN = AddUnit("dozen", _
                            dozenDelegate, _
                            0, _
                            "Groups of twelve discrete objects", _
                            "#0 Dozen")

            Dim grossDelegate As ConversionFactorProducer _
                = ConstantConverstionDelegate(1D / 144D)
            GROSS = AddUnit("gross", _
                            grossDelegate, _
                            0, _
                            "Groups of 144 discrete objects", _
                            "#0 gross")
        End Sub

        '
        ' Return the single instance of this class.
        '
        Public Shared Function getInstance() _
                            As UnitAmountFactory
            Return myInstance
        End Function

        '
        ' Return a Quantity that encapsulates the given amount and unit of
        ' measurement.
        '
        ' Parameters:
        ' - theAmount
        '   The amount to be encapsulated.
        '
        ' - theUnit
        '   The unit to be encapsulated.
        '
        ' Returns the requested Quantity.
        '
        Public Function CreateUnit( _
                            ByVal theAmount As Decimal, _
                            ByVal theUnit As Unit) As IQuantity
            Dim Unit As MeasurementUnit
            Unit = UnitToMeasurementUnit(theUnit)
            Return CreateQuantity(theAmount, Unit)
        End Function

        '
        ' Return a Quantity that encapsulates the given amount and unit of
        ' measurement.
        '
        '
        ' - theAmount
        '   The amount to be encapsulated.
        '
        ' - theUnit
        '   The unit to be encapsulated.
        '
        ' - theMaxPrecision
        '   The maximum number of decimal places to use for representing 
        ' this quantity or -1 to indicate that there is no maximum.
        '
        ' Returns the requested Quantity.
        '
        Public Function CreateUnit( _
                            ByVal theAmount As Decimal, _
                            ByVal theUnit As Unit, _
                            ByVal theMaxPrecision As Integer) _
                        As IQuantity
            Dim Unit As MeasurementUnit
            Unit = UnitToMeasurementUnit(theUnit)
            Return CreateQuantity(theAmount, Unit, _
                                    theMaxPrecision)
        End Function

        Private Function UnitToMeasurementUnit( _
                                        ByVal theUnit As Unit) _
                        As MeasurementUnit
            Select Case (theUnit)
                Case Unit.Eaches
                    Return EACHES

                Case Unit.Dozen
                    Return DOZEN

                Case Unit.Gross
                    Return GROSS
            End Select
            Dim msg As String = "Unknown unit code"
            Throw New ApplicationException(msg)
        End Function

        '
        ' Convert the given Quantity of units to the
        ' specified measurement units.
        '
        ' Parameters:
        ' - theQty
        '   The quantity to convert.
        '
        ' - theUnit
        '   The measurement unit to convert to.
        '
        ' - theMaxPrecision
        '   The maximum number of decimal places to use for representing
        ' the converted quantity or -1 to indicate no maximum.
        '
        ' Return the requested Quantity.
        '
        Public Function ConvertUnit( _
                            ByVal theQty As IQuantity, _
                            ByVal theUnit As Unit, _
                            ByVal theMaxPrecision As Integer) _
                        As IQuantity
            Dim unit As MeasurementUnit
            unit = UnitToMeasurementUnit(theUnit)
            Return MeasurementDimension.Convert(theQty, _
                                                unit, _
                                                theMaxPrecision)
        End Function

        '
        ' Convert the given Quantity of units to the
        ' specified measurement units.
        '
        ' Parameters:
        ' - theQty
        '   The quantity to convert.
        '
        ' - theUnit
        '   The measurement unit to convert to.
        '
        ' Return the requested Quantity.
        '
        Public Function ConvertUnit( _
                                ByVal theQty As IQuantity, _
                                ByVal theUnit As Unit) _
                        As IQuantity
            Dim unit As MeasurementUnit
            unit = UnitToMeasurementUnit(theUnit)
            Return MeasurementDimension.Convert(theQty, _
                                                unit, _
                                            unit.MaxPrecision)
        End Function
    End Class
End Namespace