Namespace commerce
    ' 
    ' Factory class for creating IQuantity
    ' objects that express an amount of time.
    ' 
    Public Class TimeAmountFactory
        Inherits commerce.AbstractAmountFactory

        '
        ' The single instance of this class.
        '
        Private Shared myInstance As TimeAmountFactory _
            = New TimeAmountFactory()

        '
        ' A unit for measuring milliseconds.
        '
        Private ReadOnly MILLISECOND As MeasurementUnit

        ' 
        ' A unit for measuring seconds.
        ' 
        Private ReadOnly SECOND As MeasurementUnit

        ' 
        ' A unit for measuring minutes.
        ' 
        Private ReadOnly MINUTE As MeasurementUnit

        ' 
        ' A unit for measuring hours.
        ' 
        Private ReadOnly HOUR As MeasurementUnit

        ' 
        ' A unit for measuring days.
        ' 
        Private ReadOnly DAY As MeasurementUnit

        '
        ' A unit for measuring weeks.
        '
        Private ReadOnly WEEK As MeasurementUnit

        '
        ' Constructor
        '
        Public Sub New()
            MyBase.New("Time", _
                        "This dimension for time", _
                        "second", _
                        "Seconds", _
                        -1, _
                        "#0 sec.")
            SECOND = Dimension.StandardUnit

            Dim msecDelegate As ConversionFactorProducer _
                = ConstantConverstionDelegate(1D / 1000D)
            MILLISECOND = AddUnit("millisecond", _
                                    msecDelegate, _
                                    -1, _
                                    "milliseconds", _
                                    "#0 msec.")

            Dim minDelegate As ConversionFactorProducer _
                = ConstantConverstionDelegate(60)
            MINUTE = AddUnit("minute", minDelegate, -1, "minutes", "#0 min.")

            Dim hrDelegate As ConversionFactorProducer _
                = ConstantConverstionDelegate(3600)
            HOUR = AddUnit("hour", hrDelegate, -1, "hours", "#0 hr.")

            Dim dayDelegate As ConversionFactorProducer _
                = ConstantConverstionDelegate(86400)
            DAY = AddUnit("day", dayDelegate, -1, "days", "#0 days")

            Dim wkDelegate As ConversionFactorProducer _
                = ConstantConverstionDelegate(604800)
            WEEK = AddUnit("week", wkDelegate, -1, "weeks", "#0 weeks")
        End Sub

        ' 
        ' Return the single instance of this class.
        ' 
        Public Shared Function getInstance() _
                            As TimeAmountFactory
            Return myInstance
        End Function

        '
        ' Return a Quantity that encapsulates the 
        ' given amount and unit of measurement.
        '
        ' Parameters:
        ' - theTime
        '   The amount of time to be encapsulated.
        '
        ' - theUnit
        '   The unit of measurement to be encapsulated.
        '
        ' Returns the requested Quantity.
        Public Function CreateTime(ByVal theTime As Decimal, _
                                ByVal theUnit As Time) _
                        As IQuantity
            Dim unit As MeasurementUnit
            unit = TimeToMeasurementUnit(theUnit)
            Return CreateQuantity(theTime, unit)
        End Function

        ' 
        ' Return a Quantity that encapsulates the 
        ' given amount of time and unit of measurement.
        ' 
        ' Parameters:
        ' - theAmount
        '   The amount to be encapsulated.
        '
        ' - theTime
        '   The unit of measurement to be encapsulated.
        '
        ' - theMaxPrecision
        '   The maximum number of decimal places to use for representing 
        ' this quantity or -1 to indicate there is no maximum.
        '
        ' Returns The requested Quantity.
        '
        Public Function CreateTime( _
                            ByVal theAmount As Decimal, _
                            ByVal theTime As Time, _
                            ByVal theMaxPrecision As Integer) _
                        As IQuantity
            Dim unit As MeasurementUnit
            unit = TimeToMeasurementUnit(theTime)
            Return CreateQuantity(theAmount, unit, _
                    theMaxPrecision)
        End Function

        Private Function _
                TimeToMeasurementUnit(ByVal theTime As Time) _
                As MeasurementUnit
            Select Case (theTime)
                Case Time.Millisecond
                    Return MILLISECOND

                Case Time.Second
                    Return SECOND

                Case Time.Minute
                    Return MINUTE

                Case Time.Hour
                    Return HOUR

                Case Time.Day
                    Return DAY

                Case Time.Week
                    Return WEEK
            End Select
            Dim msg As String = "Unknown time code"
            Throw New ApplicationException(msg)
        End Function

        '
        ' Convert the given Quantity of time to the
        ' specified unit of measurement.
        ' 
        ' Parameters
        ' - theQty
        '   The quantity to convert.
        '
        ' - theTime
        '   The measurement unit to convert to.
        '
        ' - theMaxPrecision
        '   The maximum number of decimal places to use for representing
        '   the converted quantity or -1 to indicate no maximum.
        ' 
        ' Returns the requested Quantity.
        '
        Public Function ConvertTime( _
                            ByVal theQty As IQuantity, _
                            ByVal theTime As Time, _
                            ByVal theMaxPrecision As Integer) _
                        As IQuantity
            Dim unit As MeasurementUnit
            unit = TimeToMeasurementUnit(theTime)
            Return MeasurementDimension.Convert(theQty, _
                                                unit, _
                                                theMaxPrecision)
        End Function

        '
        ' Convert the given Quantity of units to the
        ' specified measurement units.
        ' 
        ' Parameters:
        ' - theQty
        '   The quantity to convert.
        '
        ' - theTime
        '   The measurement unit to convert to.
        '
        ' Returns the requested Quantity.
        '
        Public Function ConvertTime( _
                            ByVal theQty As IQuantity, _
                            ByVal theTime As Time) As IQuantity
            Dim unit As MeasurementUnit
            unit = TimeToMeasurementUnit(theTime)
            Dim maxPrecision As Integer = unit.MaxPrecision
            Return MeasurementDimension.Convert(theQty, _
                                                unit, _
                                                maxPrecision)
        End Function
    End Class
End Namespace
