Namespace commerce
    '
    ' This class is a singleton that is responsible for creating 
    ' Quantity objects to represent monetary amounts.
    '
    Public Class MonetaryAmountFactory
        Inherits AbstractAmountFactory

        '
        ' The single instance of this class.
        '
        Private Shared ReadOnly myInstance As MonetaryAmountFactory _
            = New MonetaryAmountFactory()

        Private Shared ReadOnly myConverter As ICurrencyConversion _
            = New HardwiredCurrencyConversion()

        '
        ' U.S. Dollars
        '
        Private ReadOnly USD As MeasurementUnit

        '
        ' Canadian Dollars.
        '
        Private ReadOnly CAD As MeasurementUnit

        '
        ' Euros.
        '
        Private ReadOnly EUR As MeasurementUnit

        '
        ' Great Britan's Pound Sterling.
        '
        Private ReadOnly GBP As MeasurementUnit

        '
        ' Japanese Yen.
        '
        Private ReadOnly JPY As MeasurementUnit

        '
        ' Private constructor to ensure that only this class's GetInstance
        ' method can instantiate this class.
        '
        Private Sub New()
            MyBase.new("Money", _
                    "This dimension for different currencies", _
                    "USD", _
                    "U.S. Dollars", _
                    2, _
                    "$#0.00;($#0.00)")
            USD = Dimension.StandardUnit

            Dim cadDelegate As ConversionFactorProducer _
                = MoneyConversionFactorDelegate("CAN")
            CAD = AddUnit("CAN", cadDelegate, 2, _
                            "Canadian Dollars", _
                            "$#0.00;($#0.00)")

            Dim eurDelegate As ConversionFactorProducer _
                = MoneyConversionFactorDelegate("EUR")
            EUR = AddUnit("EUR", cadDelegate, 2, _
                            "Euros", _
                            "\x20AC#0.00;(\x20AC#0.00)")

            Dim gbpDelegate As ConversionFactorProducer _
                = MoneyConversionFactorDelegate("GBP")
            GBP = AddUnit("GBP", gbpDelegate, 2, _
                            "Great Britan's Pound Sterling", _
                            "\x00A3#0.00;(\x00A3#0.00)")

            Dim jpyDelegate As ConversionFactorProducer _
                = MoneyConversionFactorDelegate("JPY")
            JPY = AddUnit("JPY", jpyDelegate, 2, _
                            "Japanese Yen", _
                            "\x00A5#0.00;(\x00A5#0.00)")
        End Sub

        '
        ' Create a delegate to by used for determining the
        ' conversion rate for the given currency.
        ' 
        ' Parameters:
        ' - cur
        '   The three letter ISO 4217 abbreviation for the currency that 
        ' this object will look up exchange rates for.
        '
        ' Returns the requested delegate.
        '
        Private Function _
            MoneyConversionFactorDelegate(ByVal cur As String) _
            As ConversionFactorProducer

            Dim cnv As MoneyConversionFactor _
                = New MoneyConversionFactor(myConverter, cur)
            Return New ConversionFactorProducer(AddressOf cnv.GetValue)
        End Function

        '
        ' Return the single instance of this class.
        '
        Public Shared Function GetInstance() As MonetaryAmountFactory
            Return myInstance
        End Function

        '
        ' Return a Quantity that encapsulates the given amount and 
        ' currency.
        '
        ' Parameters:
        ' - theAmount
        '   The amount to be encapsulated.
        '
        ' - theCurrency
        '   The currency to be encapsulated.
        '
        ' Returns the requested Quantity.
        '
        Public Function CreateMoney( _
                            ByVal theAmount As Decimal, _
                            ByVal theCurrency As Currency) _
                        As IQuantity
            Dim unit As MeasurementUnit _
                = CurrencyToMeasurementUnit(theCurrency)
            Return CreateQuantity(theAmount, unit)
        End Function

        '
        ' Return a Quantity that encapsulates the given amount and 
        ' currency.
        '
        ' Parameters:
        ' - theAmount
        '   The amount to be encapsulated.
        '
        ' - theCurrency
        '   The currency to be encapsulated.
        '
        ' - theMaxPrecision
        '   The maximum number of decimal places to use for representing 
        ' this quantity or -1 to indicate that there is no maximum.
        '
        ' Returns the requested Quantity.
        Public Function _
                CreateMoney(ByVal theAmount As Decimal, _
                            ByVal theCurrency As Currency, _
                            ByVal theMaxPrecision As Integer) _
                As IQuantity
            Dim unit As MeasurementUnit _
                = CurrencyToMeasurementUnit(theCurrency)
            Return CreateQuantity(theAmount, _
                                    unit, _
                                    theMaxPrecision)
        End Function

        '
        ' Convert the given Quantity of money to the specified currency.
        '
        ' Parameters:
        ' - theQty
        '   The quantity to convert.
        '
        ' - theCurrency
        '   The currency to convert to.
        '
        ' - theMaxPrecision
        '   The maximum number of decimal places to use for representing 
        '   the converted quantity or -1 to indicate no maximum.
        '
        ' - time
        '   The effecive time for the conversion.
        '
        ' Returns the requested Quantity
        '
        Public Function _
            ConvertMoney(ByVal theQty As IQuantity, _
                        ByVal theCurrency As Currency, _
                        ByVal theMaxPrecision As Integer, _
                        ByVal time As DateTime) As IQuantity
            Dim unit As MeasurementUnit _
                = CurrencyToMeasurementUnit(theCurrency)
            Return MeasurementDimension.Convert(theQty, _
                                            unit, _
                                            theMaxPrecision, _
                                            time)
        End Function

        '
        ' Convert the given Quantity of money to the specified currency.
        '
        ' Parameters:
        ' - theQty
        '   The quantity to convert.
        '
        ' - theCurrency
        '   The currency to convert to.
        '
        ' - theMaxPrecision
        '   The maximum number of decimal places to use for representing the
        '   converted quantity or -1 to indicate no maximum.
        '
        ' Returns the requested Quantity
        Public Function _
            ConvertMoney(ByVal theQty As IQuantity, _
                            ByVal theCurrency As Currency, _
                            ByVal theMaxPrecision As Integer) _
            As IQuantity
            Return ConvertMoney(theQty, _
                                theCurrency, _
                                theMaxPrecision, _
                                DateTime.Now)
        End Function

        '
        ' Convert the given Quantity of money to the specified currency.
        '
        ' Parameters:
        ' - theQty
        '   The quantity to convert.
        '
        ' - theCurrency
        '   The currency to convert to.
        '
        ' - time
        '   The effecive time for the conversion.
        '
        ' Returns the requested Quantity.
        Public Function ConvertMoney(ByVal theQty As IQuantity, _
        ByVal theCurrency As Currency, _
        ByVal time As DateTime) As IQuantity
            Dim unit As MeasurementUnit _
                = CurrencyToMeasurementUnit(theCurrency)
            Return ConvertMoney(theQty, theCurrency, unit.MaxPrecision, time)
        End Function

        '
        ' Convert the given Quantity of money to the specified currency.
        ' 
        ' Parameters:
        ' - theQty
        '   The quantity to convert.
        '
        ' - theCurrency
        '   The currency to convert to.
        '
        ' Returns the requested Quantity.
        '
        Public Function _
                ConvertMoney(ByVal theQty As IQuantity, _
                            ByVal theCurrency As Currency) _
                As IQuantity
            Return ConvertMoney(theQty, _
                                theCurrency, _
                                (DirectCast(theQty, DecimalQuantity)).MaxPrecision, _
                                DateTime.Now)
        End Function

        Private Function CurrencyToMeasurementUnit( _
                            ByVal theCurrency As Currency) _
                        As MeasurementUnit
            Select Case (theCurrency)
                Case Currency.USDollar
                    Return USD

                Case Currency.CanadianDollar
                    Return CAD

                Case Currency.Euro
                    Return EUR

                Case Currency.Yen
                    Return JPY

                Case Currency.PoundSterling
                    Return GBP
            End Select
            Dim msg As String = "Unknown currency code"
            Throw New ApplicationException(msg)
        End Function

        '
        ' Instances of this class encapsulate a constant conversion factor.
        '
        Protected Class MoneyConversionFactor

            '
            ' The encapsulated conversion rate lookup object.
            '
            Private ReadOnly myCnv As ICurrencyConversion

            '
            ' The three letter ISO 4217 abbreviation for the currency that
            ' this object will look up exchange rates for.
            '
            Private ReadOnly myCurrencyAbbrev As String

            '
            ' Constructor
            '
            ' Parameters:
            ' - theConverter
            '   The object that this object will use to look up
            '   conversion rates for the given currency.
            '
            ' - theCurrencyAbbrev
            ' The three letter ISO 4217 abbreviation for the currency that
            ' this object will look up exchange rates for.
            '
            Public Sub New( _
                    ByVal theConverter As ICurrencyConversion, _
                    ByVal theCurrencyAbbrev As String)
                myCnv = theConverter
                myCurrencyAbbrev = theCurrencyAbbrev
            End Sub

            ' 
            ' Return the value encapsulated by this object.
            ' 
            ' Parameters:
            ' - args
            '   A one element object array whose single element contains a
            '   DateTime specifying an effective date for the desired 
            '   conversion rate.
            '
            ' Returns the value encapsulated by this object.
            '
            Public Function GetValue(ByVal args() As Object) As Decimal
                Return myCnv.ConverstionToUSDollars( _
                                                myCurrencyAbbrev, _
                                                DirectCast(args(0), DateTime))
            End Function
        End Class
    End Class
End Namespace

