﻿using UnityEngine;

// Skrypt modyfikuje położenie kamery, tak by zawsze odpowiadało ono
// położeniu (w pionie, współrzędna Y) innego obiektu; dopuszczalny 
// zakres przesuwania kamery jest ograniczony.
public class CameraFollow : MonoBehaviour {

	// Obiekt, do którego ma być dostosowywane położenie kamery
	public Transform target;

	// Najwyższy punkt, w którym może się znaleźć kamera
	public float topLimit = 10.0f;

	// Najniższy punkt, w którym może się znaleźć kamera
	public float bottomLimit = -10.0f;

	// Szybkość przesuwania kamery do punktu docelowego
	public float followSpeed = 0.5f;

	// Kiedy położenie wszystkich obiektów zostanie już zaktualizowane,
	// możemy zacząć przesuwać kamerę do punktu docelowego
	void LateUpdate () {

		// Jeśli mamy jakiś element, który kamera ma śledzić...
		if (target != null) {

			// Pobieramy jego położenie
			Vector3 newPosition = this.transform.position;

			// Określamy gdzie kamera powinna się znaleźć
			newPosition.y = Mathf.Lerp (
				newPosition.y, 
				target.position.y, followSpeed);

			// Dostosowujemy miejsce docelowe do dopuszczalnego zakresu
			// położenia kamery
			newPosition.y = Mathf.Min(newPosition.y, topLimit);
			newPosition.y = Mathf.Max(newPosition.y, bottomLimit);

			// Aktualizujemy położenie kamery
			transform.position = newPosition;
		}

	}

	// Po wybraniu w edytorze, rysujemy linię od górnego do dolnego 
	// dopuszczalnego położenia kamery
	void OnDrawGizmosSelected() {
		Gizmos.color = Color.yellow;

		Vector3 topPoint = new Vector3(
			this.transform.position.x, 
			topLimit, this.transform.position.z);
		Vector3 bottomPoint = new Vector3(
			this.transform.position.x, 
			bottomLimit, this.transform.position.z);

		Gizmos.DrawLine(topPoint, bottomPoint);
	}
}
