#
# Wycena europejskich opcji kupma 
# w modelu BSM (1973)
#
# (c) Dr Yves J. Hilpisch
# Uczenie przez wzmacnianie w branży finansowej
#

from math import log, sqrt, exp
from scipy import stats


def bsm_call_value(St, K, T, t, r, sigma):
    ''' Wycena europejskich opcji kupna według modelu BSM.
    Wzór analityczny.

    Parametery
    ==========
    St: float
        Poziom cen akcji lub indeksu w dniu lub czasie t.
    K: float
        Stała cena wykonania.
    T: float
        Dzień lub czas wygasania opcji (w ułamkach lat).
    t: float
        Bieżący dzień lub czas.
    r: float
        Stała krótkoterminowa stopa wolna od ryzyka.
    sigma: float
        Współczynnik zmienności w członie dyfuzji.

    Zwracane wartości
    =======
    value: float
        Obecna wartość europejskiej opcji kupna.
    '''
    St = float(St)
    d1 = (log(St / K) + (r + 0.5 * sigma ** 2) * (T - t)) / (sigma * sqrt(T - t))
    d2 = (log(St / K) + (r - 0.5 * sigma ** 2) * (T - t)) / (sigma * sqrt(T - t))
    # stats.norm.cdf --> dystrybuanta dla rozkładu normalnego.
    value = (St * stats.norm.cdf(d1, 0, 1) -
             K * exp(-r * (T - t)) * stats.norm.cdf(d2, 0, 1))
    return value

