import { expect } from 'chai';
import sinon from 'sinon';
import fetchPonyfill from 'fetch-ponyfill';
const { Response, Headers } = fetchPonyfill();
import * as errorTypes from '../reducers/errorTypes';
import FetchSpeakerService from './fetchSpeakerService';

describe('Usługa fetch prelegentów', () => {
  it('istnieje', () => {
    expect(FetchSpeakerService).to.exist;
  });

  it('może być skonstruowana', () => {
    // arrange
    let service = new FetchSpeakerService();

    // assert
    expect(service).to.be.an.instanceof(FetchSpeakerService);
  });

  describe('Po inicjalizacji', () => {
    let service = null;
    let fetch = null;

    beforeEach(() => {
      fetch = sinon.stub(global, 'fetch');
      service = new FetchSpeakerService('http://localhost');
    });

    afterEach(() => {
      fetch.restore();
    });

    describe('Create', () => {
      it('istnieje', () => {
        expect(service.create).to.exist;
      });
    });

    describe('getAll', () => {
      it('istnieje', () => {
        // assert
        expect(service.getAll).to.exist;
      });

      describe('Żaden prelegent nie istnieje', () => {
        it('zwraca pustą tablicę', () => {
          // arrange
          fetch.returns(okResponse([]));

          // act
          let promise = service.getAll();

          // assert
          return promise.then(result => {
            expect(result).to.have.lengthOf(0);
          });
        });
      });

      describe('Lista prelegentów', () => {
        it('zwraca prelegentów', () => {
          // arrange
          fetch.returns(okResponse([{}]));

          // act
          let promise = service.getAll();

          // assert
          return promise.then(result => {
            expect(result).to.have.lengthOf(1);
          });
        });
      });
    });

    describe('Pobieranie prelegenta po Id', () => {
      it('istnieje', () => {
        // assert
        expect(service.getById).to.exist;
      });

      describe('Prelegent nie istnieje', () => {
        it('generowany jest błąd SPEAKER_NOT_FOUND', () => {
          // arrange
          fetch.returns(notFoundResponse());

          // act
          const promise = service.getById('nieistniejacy-prelegent');

          // assert
          return promise
            .then(() => {
              throw { type: 'Błąd nie został zwrócony' };
            })
            .catch(error => {
              expect(error.type).to.equal(errorTypes.SPEAKER_NOT_FOUND);
            });
        });
      });

      describe('Prelegent istnieje', () => {
        it('zwraca prelegenta', () => {
          // arrange
          const speaker = {
            id: 'testowy-prelegent'
          };
          fetch.returns(okResponse(speaker));

          // act
          let promise = service.getById('testowy-prelegent');

          // assert
          return promise.then(speaker => {
            expect(speaker).to.not.be.null;
            expect(speaker.id).to.equal('testowy-prelegent');
          });
        });
      });
    });
  });
});

function baseResponse() {
  let response = new Response();
  response.headers = new Headers({
    'Content-Type': 'application/json'
  });
  response.ok = true;
  response.status = 200;
  response.statusText = 'OK';

  return response;
}

function okResponse(body) {
  return new Promise((resolve, reject) => {
    let response = baseResponse();
    response.body = JSON.stringify(body);

    resolve(response);
  });
}

function notFoundResponse() {
  return new Promise((resolve, reject) => {
    let response = baseResponse();
    response.ok = false;
    response.status = 404;
    response.statusText = 'NOT FOUND';

    resolve(response);
  });
}
