// Przykładowy kod do postu na blogu zatytułowanego 'Obiekty oczekujące'
//
// Prawa autorskie (c) Lewis Baker

#include <coroutine>
#include <atomic>

class async_manual_reset_event
{
public:

  async_manual_reset_event(bool initiallySet = false) noexcept;

  // Brak kopiowania/przenoszenia
  async_manual_reset_event(const async_manual_reset_event&) = delete;
  async_manual_reset_event(async_manual_reset_event&&) = delete;
  async_manual_reset_event& operator=(const async_manual_reset_event&) = delete;
  async_manual_reset_event& operator=(async_manual_reset_event&&) = delete;

  bool is_set() const noexcept;

  struct awaiter;
  awaiter operator co_await() const noexcept;

  void set() noexcept;
  void reset() noexcept;

private:

  friend struct awaiter;

  // - 'this' => ustawienie stanu
  // - w przeciwnym razie => nie ustawiony (początek listy dowiązanej obiektu awaiter*).
  mutable std::atomic<void*> m_state;

};

struct async_manual_reset_event::awaiter
{
  awaiter(const async_manual_reset_event& event) noexcept
  : m_event(event)
  {}

  bool await_ready() const noexcept;
  bool await_suspend(std::coroutine_handle<> awaitingCoroutine) noexcept;
  void await_resume() noexcept {}

private:
  friend struct async_manual_reset_event;

  const async_manual_reset_event& m_event;
  std::coroutine_handle<> m_awaitingCoroutine;
  awaiter* m_next;
};

bool async_manual_reset_event::awaiter::await_ready() const noexcept
{
  return m_event.is_set();
}

bool async_manual_reset_event::awaiter::await_suspend(
  std::coroutine_handle<> awaitingCoroutine) noexcept
{
  // Specjalna wartość m_state wskazująca, że zdarzenie ma stan ustawienia.
  const void* const setState = &m_event;

  // Ukrywanie uchwytu współprogramu oczekującego.
  m_awaitingCoroutine = awaitingCoroutine;

  // Próba umieszczenia w sposób atomowy tego obiektu awaiter na początku listy.
  void* oldValue = m_event.m_state.load(std::memory_order_acquire);
  do
  {
    // Natychmiastowe wznawianie, jeśli istnieje już stan ustawienia.
    if (oldValue == setState) return false; 

    // Aktualizowanie listy dowiązanej w celu wskazania bieżącego początku.
    m_next = static_cast<awaiter*>(oldValue);

    // Podjęcie na koniec próby zamiany starego początku listy przez
	// wstawienie tego obiektu awaiter jako nowego początku listy.
  } while (!m_event.m_state.compare_exchange_weak(
             oldValue,
             this,
             std::memory_order_release,
             std::memory_order_acquire));

  // Pomyślnie umieszczono w kolejce. Utrzymano stan wstrzymania.
  return true;
}

async_manual_reset_event::async_manual_reset_event(
  bool initiallySet) noexcept
: m_state(initiallySet ? this : nullptr)
{}

bool async_manual_reset_event::is_set() const noexcept
{
  return m_state.load(std::memory_order_acquire) == this;
}

void async_manual_reset_event::reset() noexcept
{
  void* oldValue = this;
  m_state.compare_exchange_strong(oldValue, nullptr, std::memory_order_acquire);
}

void async_manual_reset_event::set() noexcept
{
  // Wymaga stanu release, aby kolejny operator co_await miał dostęp do
  // poprzednich operacji zapisu.
  // Wymaga stanu acquire, aby współprogramy oczekujące miały dostęp do
  // poprzednich operacji zapisu.
  void* oldValue = m_state.exchange(this, std::memory_order_acq_rel);
  if (oldValue != this)
  {
    // Nie było jeszcze stanu ustawienia.
	// Potraktuj starą wartość jako początek listy dowiązanej już
	// uzyskanych obiektów oczekujących, które wymagają wznowienia.
    auto* waiters = static_cast<awaiter*>(oldValue);
    while (waiters != nullptr)
    {
      // Wczytanie m_next przed wznowieniem współprogramu, gdyż
	  // operacja ta prawdopodobnie spowoduje usunięcie obiektu awaiter.
      auto* next = waiters->m_next;
      waiters->m_awaitingCoroutine.resume();
      waiters = next;
    }
  }
}

async_manual_reset_event::awaiter
async_manual_reset_event::operator co_await() const noexcept
{
  return awaiter{ *this };
}

// Prosta klasa zadania na potrzeby współprogramów nie zwracających wartości.
struct task
{
	struct promise_type
	{
		task get_return_object() { return {}; }
		std::suspend_never initial_suspend() { return {}; }
		std::suspend_never final_suspend() noexcept { return {}; }
		void return_void() {}
		void unhandled_exception() {}
	};
};

task example(async_manual_reset_event& event)
{
	co_await event;
}

int main() {
    async_manual_reset_event ev;
    example(ev);
}
