--
-- SymfonyYmlExport V1.03
--
-- Mysql Workbench plugin for processing schema into symfony's yml format.
--
-- Copyright Jason Rowe <jason.rowe@milestoneip.com> / MilestoneIP 2008 - Milestone IP LTD <www.milestoneip.com>
--
-- Version 1.03
--
-- This plugin was inspired by the great work in the PropelExport plugin written by
-- Daniel Haas <daniel.haas@cn-consult.eu> Thank you Daniel, i learned lua from deciphering your source code.
--
-- I have tried to make this work as generically as possible and produce the most short hand syntax
-- available in the schema yml files.
--
-- Notes:
-- automatic primary keys. Whenever an integer primary key is used without any other 'modifiers' we use the ~ short hand syntax.
-- automatic created/updated columns. whenever these columns are added to the schema we use the ~ short hand syntax.
-- we use the shorthand syntax for foreign keys adding foreignTable, foreignReference & onDelete into the actual column
--
-- please report any bugs to <jason.rowe@milestoneip.com>.
-- please report any change requests to <jason.rowe@milestoneip.com>
--
-- For more information about symfony, please goto <http://www.symfony-project.org>
--
--
-- The module is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
-- See the GNU General Public License for more details.
--
-- We deliver this software for free, and expect no return other than to improve the usefulness of
-- symfony together with mysql workshop.
--
-- Support:
-- Please note that while i will try my best to answer your queries and emails, I work full time on other
-- projects and cannot answer emails immediately.
--
-- I hope you find the software useful and it speeds up development for you.
--
-- Jason Rowe 2008

function getModuleInfo()
	return {
		name= "SymfonyYmlExport",
		author= "MySQL AB.",
		version= "1.0",
		implements= "PluginInterface",
		functions= {
		"getPluginInfo:l<o@app.Plugin>:",
		"exportSymfonyYmlToClipboard:i:o@db.Catalog",
        "exportSymfonyYmlToFile:i:o@db.Catalog"
		}
	}
end


-- helper function to create a descriptor for an argument of a specific type of object
function objectPluginInput(type)
	return grtV.newObj("app.PluginObjectInput", {objectStructName= type})
end

function getPluginInfo()
    -- create the list of plugins that this module exports
	local l
    local plugin

    -- create the list of plugins that this module exports
	l= grtV.newList("object", "app.Plugin")
	
    plugin= grtV.newObj("app.Plugin", {
		name= "wb.util.exportSymfonyYmlToClipboard",
        caption= "Copy Symfony Yml to Clipboard",
		moduleName= "SymfonyYmlExport",
		pluginType= "normal", 
		moduleFunctionName= "exportSymfonyYmlToClipboard",
		inputValues= {objectPluginInput("db.Catalog")},
		rating= 100, 
		showProgress= 0,
		groups= {"Catalog/Utilities", "Menu/Catalog"}
	})
    -- fixup owner
    plugin.inputValues[1].owner= plugin

    -- add to the list of plugins
    grtV.insert(l, plugin)
    
    
    plugin= grtV.newObj("app.Plugin", {
		name= "wb.util.exportSymfonyYmlToFile",
        caption= "Export Symfony Yml to File",
		moduleName= "SymfonyYmlExport",
		pluginType= "normal", 
		moduleFunctionName= "exportSymfonyYmlToFile",
		inputValues= {objectPluginInput("db.Catalog")},
		rating= 100, 
		showProgress= 0,
		groups= {"Catalog/Utilities", "Menu/Catalog"}
	})
    -- fixup owner
    plugin.inputValues[1].owner= plugin

    -- add to the list of plugins
    grtV.insert(l, plugin)

	return l
end

--
-- object definitions
--
sfTableColumn = {}
function sfTableColumn:new() -- The constructor
    local object = {
        name = nil,
        required = false,
        primaryKey = false,
        autoIncrement = false,
        type = nil,
        length = 0,
        default = nil,
        sequence = nil,
        index = false,
        foreignTable = nil,
        foreignReference = nil,
        onDelete = nil,
        isCulture = false
    }
    setmetatable(object, {
        -- Overload the index event so that fields not present within the object are
        -- looked up in the prototype Vector table
        __index = sfTableColumn
    })
    return object
end

function sfTableColumn:printYml ()
    local yml
    local sep = ", "
    
    if (self.name ~= nil)
    then
        -- self.name == "id" (only if the current type is integer - else we need to specifically specify the identifier.)
        if (self.name == "id" and self.type == "integer")
        then
            return "~"
        end
        if (self.name == "created_at" or self.name == "created_on" or self.name == "updated_at" or self.name == "updated_on")
        then
            return "~"
        end
    end
    
    -- print(self.name)
    -- print(self.type)
    -- print(self.required)
    -- print(self.primaryKey)
    -- print(self.autoIncrement)
    -- print(self.length)
    -- print(self.default)
    -- print(self.sequence)
    -- print(self.index)
    -- print(self.foreignTable)
    -- print(self.foreignReference)
    -- print(self.onDelete)
    -- print(self.isCulture)
    -- print("")
    
    -- short syntax if only a type is specified
    if (self.type ~= nil
        and self.required == false
        and self.primaryKey == false
        and self.autoIncrement == false
        and self.length == 0
        and self.default == nil
        and self.sequence == nil
        and self.index == false
        and self.foreignTable == nil
        and self.foreignReference == nil
        and self.onDelete == nil
        and self.isCulture == false
    )
    then
        return self.type       
    end
    
    -- type
    if (self.length == 0)
    then
        yml = "type: " .. self.type .. sep
    else
        yml = "type: " .. self.type .. "(" .. self.length .. ")" .. sep
    end
    
    if (self.required == true)
    then
        yml = yml .. "required: true" .. sep
    end
    
    if (self.primaryKey == true)
    then
        yml = yml .. "primaryKey: true" .. sep
    end
    
    if (self.autoIncrement == true)
    then
        yml = yml .. "autoIncrement: true" .. sep
    end
    
    if (self.default ~= nil)
    then
        yml = yml .. "default: " .. self.default .. sep
    end
    
    if (self.sequence ~= nil)
    then
        yml = yml .. "sequence: " .. self.sequence .. sep
    end
    
    if (self.index == true)
    then
        yml = yml .. "index: true" .. sep
    end
    
    if (self.onDelete ~= nil)
    then
        yml = yml .. "onDelete: " .. self.onDelete .. sep
    end
    
    if (self.foreignTable ~= nil)
    then
        yml = yml .. "foreignTable: " .. self.foreignTable .. sep
    end
    
    if (self.foreignReference ~= nil)
    then
        yml = yml .. "foreignReference: " .. self.foreignReference .. sep
    end
    
    if (self.isCulture == true)
    then
        yml = yml .. "isCulture: true" .. sep
    end

    
    yml = string.sub(yml, 1, -3)
    
    return "{ " .. yml .. " }"
end


--    
-- implementation
--

function exportSymfonyYmlToClipboard(cat)

	local yml = convertSchemaIntoSymfonyYml(cat)
	
    Workbench:copyToClipboard(yml)

    return 0
end

function exportSymfonyYmlToFile(cat)

	local yml = convertSchemaIntoSymfonyYml(cat)

    if (cat.customData["symfonyYmlExportPath"] ~= nil)
    then
        -- print("\nFilepath is: "..cat.customData["symfonyYmlExportPath"]);
        if (Workbench:confirm("Proceed?","Do you want to overwrite previously exported file "..cat.customData["symfonyYmlExportPath"].." ?") == 1)
        then
            symfonyYmlExportPath=cat.customData["symfonyYmlExportPath"];
        else
            symfonyYmlExportPath=Workbench:input('Filename? Please enter Filename to export the symfony schema to');
            if (symfonyYmlExportPath~="")
            then
                -- Try to save the filepath for the next time:
                cat.customData["symfonyYmlExportPath"]=symfonyYmlExportPath;
            end
        end
    else
        symfonyYmlExportPath=Workbench:input('Filename? Please enter Filename to export the symfony schema to');
        if (symfonyYmlExportPath~="")
        then
            -- Try to save the filepath for the next time:
            cat.customData["symfonyYmlExportPath"]=symfonyYmlExportPath;
        end
    end
  
    if symfonyYmlExportPath~='' then
        f = io.open(symfonyYmlExportPath,"w");
        if (f~=nil) then
            f.write(f,yml);
            f.close(f);
            print('\n > Symfony-Schema was exported to '..symfonyYmlExportPath);  
        else
            print('\n > Could not open file '..symfonyYmlExportPath..'!');
        end
    else
        print('\n > Symfony-Schema was NOT exported as no path was given!');
    end
    
    return 0
end



-- base function for processing the mysql workbench model into symfonys yml format
function convertSchemaIntoSymfonyYml(cat)

	local newline = "\n"
    local separator = ":"
	local tab = "  "
    local tab2 = tab .. tab
    local tab3 = tab2 .. tab
    local tab4 = tab3 .. tab
    local yml = "propel" .. separator .. newline
	local i
	local j
	local k
	local z
	local tab_table = tab
	local tab_column = tab .. tab
    local maxColumnLength = getLongestColumnLength(cat)
    local tableIndexYml
    local columnIndexYml
    
    -- new column processor
    local tableColumns
    
    -- multiple schemas
	for i = 1, grtV.getn(cat.schemata) do
    
        schema = cat.schemata[i]
		
		-- multiple tables
		for j = 1, grtV.getn(schema.tables) do
		
			table = schema.tables[j]
            
            -- debug
            if (table.name == "test")
            then
                --print(table)
            end
            
			yml = yml .. tab_table .. table.name .. separator .. newline
			
			-- multiple table columns
            --
            -- Items to handle
            -- name - done
            -- type - done
            -- required - done
            -- primaryKey - done
            -- autoIncrement - done
            -- length - done
            -- default - done
            -- sequence - ignore for mysql
            -- index                                    - done via long hand notation
            -- foreignTable
            -- foreignReference
            -- onDelete
            -- isCulture

			for k = 1, grtV.getn(table.columns) do
			
				column = table.columns[k]
				
                currentColumn = nil
                currentColumn = sfTableColumn:new()
				currentColumn.name = column.name
                currentColumn.type = simpleType2SymfonyPropelDatatype(column.simpleType)
                
				--print(column)
                --print(column.flags)
                
                -- size
                if (column.length ~= nil)
                then
                    if (column.length == -1)
                    then
                       currentColumn.length = 0
                    else
                       currentColumn.length = column.length
                    end
                end
                
                -- required
                currentColumn.required = column.isNotNull == 1
        
                -- autoInc					
                currentColumn.autoIncrement = column.autoIncrement == 1
                
                -- default value
                if (column.defaultValue ~= nil and column.defaultValue ~= "")
                then
                    if (string.find(column.defaultValue,"'") ~= nil)
                    then
                        currentColumn.default = string.sub(column.defaultValue, 2, -2)
                    else                    
                        currentColumn.default = column.defaultValue
                    end
                    
                end
                
                -- primaryKey (now using index lookup)
                if (#table.indices > 0)
                then
                    for z = 1, grtV.getn(table.indices) do
                        index = table.indices[z]
                        --print(index)
                        if (index.indexType=="PRIMARY")
                        then
                            for l=1, grtV.getn(index.columns) do
                                indexColumn = index.columns[l]
                                if (indexColumn.referencedColumn ~= nil)
                                then
                                    referencedColumn = indexColumn.referencedColumn
                                    --print("referencedColumn")
                                    --print(referencedColumn)
                                    if (referencedColumn.name == column.name)
                                    then
                                        currentColumn.primaryKey = true;
                                    end
                                end
                            end
                        end
                    end		
                end
                
                -- foreign key
                if (#table.foreignKeys > 0)
                then
                    for z = 1, grtV.getn(table.foreignKeys) do
                        foreignKey = table.foreignKeys[z]
                        
                        --if(#foreignKey.columns and #foreignKey.referencedColumns)
                        --then
                            --print("--" .. column.name .. " - " .. foreignKey.name)
                            --print(foreignKey)
                            --print("")
                            --print("")
                            
                            foreignKeyColumn = foreignKey.columns[1]
                            foreignKeyReferencedColumn = foreignKey.referencedColumns[1]                            
                            
                            if (foreignKeyColumn.name == column.name)
                            then                                
                                
                                if (foreignKey.referencedTable ~= nil)
                                then
                                    currentColumn.foreignTable = foreignKey.referencedTable.name
                                    currentColumn.foreignReference = foreignKeyReferencedColumn.name 
                                end
                                
                                
                                if(foreignKey.deleteRule == "SET NULL")
                                then
                                    currentColumn.onDelete = "null"
                                elseif (foreignKey.deleteRule == "CASCADE")
                                then
                                    currentColumn.onDelete = "cascade"
                                end
                                
                            end
                       
                        --end
                    end		
                end
                
				yml = yml .. tab_column .. padRight(column.name .. separator, maxColumnLength + 8, " ") .. tab .. currentColumn:printYml() .. newline
			end
            
            -- index
            tableIndexYml = nil
            if (#table.indices > 0)
            then
                for z = 1, grtV.getn(table.indices) do
                    index = table.indices[z]
                    --print(index)
                    if (index.indexType=="INDEX")
                    then
                        if (tableIndexYml == nil)
                        then
                            tableIndexYml = ""
                        end
                        
                        columnIndexYml = ""
                    
                        for l=1, grtV.getn(index.columns) do
                            column = index.columns[l]
                            columnIndexYml = columnIndexYml .. column.referencedColumn.name .. ", "
                        end
                        
                        tableIndexYml = tableIndexYml .. tab_column .. tab .. padRight(index.name .. separator, maxColumnLength + 8, " ") .. "[ " .. string.sub(columnIndexYml, 1, -3) .. " ]" .. newline
                    end
                end		
            end
            
            if (tableIndexYml ~= nil)
            then
                yml = yml .. tab_column .. "_indexes:" .. newline
                yml = yml .. tableIndexYml
            end
            
            -- unique (code duplication) - should handle this better
            tableIndexYml = nil
            if (#table.indices > 0)
            then
                for z = 1, grtV.getn(table.indices) do
                    index = table.indices[z]
                    --print(index)
                    if (index.indexType=="UNIQUE")
                    then
                        if (tableIndexYml == nil)
                        then
                            tableIndexYml = ""
                        end
                        
                        columnIndexYml = ""
                    
                        for l=1, grtV.getn(index.columns) do
                            column = index.columns[l]
                            columnIndexYml = columnIndexYml .. column.referencedColumn.name .. ", "
                        end
                        
                        tableIndexYml = tableIndexYml .. tab_column .. tab .. padRight(index.name .. separator, maxColumnLength + 8, " ") .. "[ " .. string.sub(columnIndexYml, 1, -3) .. " ]" .. newline
                    end
                end		
            end
            
            if (tableIndexYml ~= nil)
            then
                yml = yml .. tab_column .. "_uniques:" .. newline
                yml = yml .. tableIndexYml
            end
            
            -- foreign keys (long hand syntax)
--  tableForeignKeyYml = nil
--  if (#table.foreignKeys > 0)
--  then
--      tableForeignKeyYml = ""
--      for k = 1, grtV.getn(table.foreignKeys) do
--          foreignKey = table.foreignKeys[k]
--          --print(foreignKey)
--          tableForeignKeyYml = tableForeignKeyYml .. tab3 .. foreignKey.name .. separator .. newline
--          tableForeignKeyYml = tableForeignKeyYml .. tab4 .. padRight("foreignTable" .. separator, maxColumnLength + 6, " ") .. foreignKey.referencedTable.name .. newline
--          
--          -- onDelete
--          if (foreignKey.deleteRule == "CASCADE")
--          then
--              tableForeignKeyYml = tableForeignKeyYml .. tab4 .. padRight("onDelete" .. separator, maxColumnLength + 6, " ") .. "cascade" .. newline
--          elseif (foreignKey.deleteRule == "SET NULL")
--          then
--              tableForeignKeyYml = tableForeignKeyYml .. tab4 .. padRight("onDelete" .. separator, maxColumnLength + 6, " ") .. "null" .. newline
--          end
--          tableForeignKeyYml = tableForeignKeyYml .. tab4 .. "references" .. separator .. newline
--          tableForeignKeyYml = tableForeignKeyYml .. tab4 .. " - { local: " .. foreignKey.columns[1].name .. ", foreign: " .. foreignKey.referencedColumns[1].name .. " }" .. newline
--      end  
--  end
--  
--  if (tableForeignKeyYml ~= nil)
--  then
--      yml = yml .. tab_column .. "_foreignKeys:" .. newline
--      yml = yml .. tableForeignKeyYml
--  end
			
			
        end
    end
	
    --print("")
    print("#-------------------------")
    --print("")
    print(yml)
	return yml
    
    
end

-- find out how long the longest column in the whole schema is
function getLongestColumnLength(cat)

    local maxColumnLength = 0
    local schema
    local table
    local column
    
	for i = 1, grtV.getn(cat.schemata) do	
        schema = cat.schemata[i]
        
        -- do table columns
		for j = 1, grtV.getn(schema.tables) do
			table = schema.tables[j]
			for k = 1, grtV.getn(table.columns) do
			
				column = table.columns[k]
				if(maxColumnLength < string.len(column.name))
                then
                    maxColumnLength = string.len(column.name)
                end
            end
            
            -- now check the length of indicies
            for m = 1, grtV.getn(table.indices) do
                index = table.indices[m]
                
                if (index.indexType ~= "FOREIGN")
                then
                    if(maxColumnLength < string.len(index.name))
                    then
                        maxColumnLength = string.len(index.name)
                    end
                end
                
            end
        end  
    end

	return maxColumnLength
end

function padRight(s, length, padchar)
    local i    
    if (string.len(s) >= length)
    then
        return s
    end    
    i = length - string.len(s)
    return s .. string.rep(padchar, i)    
end


-- convert simple type to symfony/propel types
function simpleType2SymfonyPropelDatatype(simpleType)

  -- local propelType="**UNKNOWN** ("..simpleType.name..")"
  -- We assume that the simpleType corresponds to the propel type by default
  -- This is correct 95% of the time
  if (simpleType~=nil) then
	  
	  local propelType=simpleType.name
	  
	  -- convert INT to INTEGER
	  if (simpleType.name=="INT") then
		propelType = "integer"
	  end
	  
	  -- convert text types to CLOBs
	  if (simpleType.name=="TINYTEXT" or simpleType.name=="VARCHAR") then
		propelType = "varchar"
	  end
      
	  if (simpleType.name=="TEXT") then
		propelType = "longvarchar"
	  end
	  if (simpleType.name=="MEDIUMTEXT") then
		propelType = "clob"
	  end
	  if (simpleType.name=="LONGTEXT") then
		propelType = "clob"
	  end

	  return string.lower(propelType)
  else 
    return "UNKNOWN"
  end
end

