import java.util.*;

/** Klasa KWPriorityQueue implementuje interfejs Queue,
*   korzystajc ze stery na obiekcie ArrayList. Wykorzystuje si stert,
*   by najmniejszy element zawsze znajdowa si na szczycie.
*   @author Koffman and Wolfgang
*/

public class KWPriorityQueue < E >
    extends AbstractQueue < E >
    implements Queue < E > {

  // pola danych
  /** Obiekt ArrayList przechowujcy dane. */
  private ArrayList < E > theData;

  /** Opcjonalna referencj do obiektu Comparator. */
  Comparator < E > comparator = null;

  // metody
  // konstruktor
  public KWPriorityQueue() {
    theData = new ArrayList < E > ();
  }

  /** Tworzy kolejk priorytetow bazujc na stercie o pocztkowej
       pojemnoci, ktra do sortowania elementw uywa
       przekazanego komparatora.
       @param cap Pocztkowa pojemno kolejki priorytetowej.
       @param comp Komparator uywamy do porwnywania elementw.
       @throws WyjtekIllegalArgumentException, jeli cap jest mniejsze od 1.
   */
  public KWPriorityQueue(int cap, Comparator < E > comp) {
    if (cap < 1)
      throw new IllegalArgumentException();
    theData = new ArrayList < E > (cap + 1);
    comparator = comp;
  }

  /** Wstawia element do kolejki priorytetowej.
      pocz: Tablica theData zawiera poprawne dane sterty.
      koc: Element znajduje si w kolejce priorytetowej
            a theData zawiera poprawne dane sterty.
      @param item Wstawiany element.
      @throws Wyjtek NullPointerException, jeli element do wstawienia wynosi null.
   */
  public boolean offer(E item) {
    // Dodaje element do sterty.
    theData.add(item);
    // Dzieckiem jest nowowstawiony element.
    int child = theData.size() - 1;
    int parent = (child - 1) / 2; // Znajduje rodzica dziecka.
    // Wykonanie poprawnej sterty.
    while (parent >= 0 && compare(theData.get(parent),
                                  theData.get(child)) > 0) {
      swap(parent, child);
      child = parent;
      parent = (child - 1) / 2;
    }
    return true;
  }

  /** Usuwa element z kolejki priorytetowej.
      pocz: Tablica theData zawiera poprawne dane sterty.
      koc: Zosta usunity najmniejszy element.
            Tablica theData zawiera poprawne dane sterty.
      @return Element o najmniejszej wartoci priorytetu lub null, jeli stos jest pusty.
   */
  public E poll() {
    if (isEmpty()) {
      return null;
    }
    // Zapamitaj szczyt sterty.
    E result = theData.get(0);
    // Jeli to jedyny element, po prostu go usu.
    if (theData.size() == 1) {
      theData.remove(0);
      return result;
    }
    /* Usuwa ostatni element obiektu ArrayList i umieszcza go na pierwszej
       pozycji. */
    theData.set(0, theData.remove(theData.size() - 1));
    // Rodzic zaczyna si od szczytu.
    int parent = 0;
    while (true) {
      int leftChild = 2 * parent + 1;
      if (leftChild >= theData.size()) {
        break; // Poza stert.
      }
      int rightChild = leftChild + 1;
      int minChild = leftChild; // Za, i leftChild jest mniejszy.
      // Sprawd, czy rightChild jest mniejszy.
      if (rightChild < theData.size()
          && compare(theData.get(leftChild),
                     theData.get(rightChild)) > 0) {
        minChild = rightChild;
      }
      // asercja: minChild to indeks najmniejszego dziecka.
      // Przesu mniejsze dziecko na gr.
      if (compare(theData.get(parent),
                  theData.get(minChild)) > 0) {
        swap(parent, minChild);
        parent = minChild;
      }
      else { // Udao si odtworzy poprawn stert.
        break;
      }
    }
    return result;
  }

  /** Porwnuje dwa obiekty, uywajc metody compare() obiektu Comparator
       lub porzdku naturalnego, wywoujc metod compareTo().
       pocz: Jeli comparator wynosi null, left i right implementuje interfejs Comparable<E>.
       @param left Jeden element.
       @param right Drugi element.
       @return Ujemn warto cakowit, jeli left jest mniejsze od right,
          0, jeli left jest rwne right,
          dodatni warto cakowit, jeli left > right.
       @throws Wyjtek ClassCastException, jeli elementy nie implementuj interfejsu Comparable.
   */
  private int compare(E left, E right) {
    if (comparator != null) { // Zdefiniowano obiekt Comparator.
      return comparator.compare(left, right);
    }
    else { // Uyj metody compareTo() obiektu left.
      return ( (Comparable < E > ) left).compareTo(right);
    }
  }

/**** WICZENIE ****/
}
