import java.util.*;
import java.io.*;

/** Klasa reprezentujca i generujca drzewo Huffmana.
*   @author Koffman and Wolfgang
* */

public class HuffmanTree
    implements Serializable {

  // klasy zagniedone
  /** Element danych drzewa Huffmana. */
  public static class HuffData
      implements Serializable {
    // pola danych
    /** Waga przypisana danemu obiektowi HuffData. */
    private double weight;

    /** Symbol z alfabetu, jeli obiekt jest liciem. */
    private Character symbol;

    public HuffData(double weight, Character symbol) {
      this.weight = weight;
      this.symbol = symbol;
    }
  }

  // pola danych
  /** Referencja do utworzonego drzewa Huffmana. */
  private BinaryTree < HuffData > huffTree;

  /** Klasa zagniedona okrelajca komparator drzew Huffmana. */
  private static class CompareHuffmanTrees
      implements Comparator < BinaryTree < HuffData >> {
    /** Oprwnuje dwa obiekty.
        @param left Lewa strona porwnywania.
        @param right Prawa strona porwnywania.
        @return -1, jeli left < right; 0, jeli left == right;
                +1, jeli left > right.
     */
    public int compare(BinaryTree < HuffData > treeLeft,
                       BinaryTree < HuffData > treeRight) {
      double wLeft = treeLeft.getData().weight;
      double wRight = treeRight.getData().weight;
      return Double.compare(wLeft, wRight);
    }
  }

  /** Buduje drzewo Huffmana na podstawie przekazanych wag i symboli.
      koc:  Referencja huffTree wskazuje na pene drzewo Huffmana.
      @param symbols Tablica obiektw huffData.
   */
  public void buildTree(HuffData[] symbols) {
    Queue < BinaryTree < HuffData >> theQueue
        = new PriorityQueue < BinaryTree < HuffData >>
        (symbols.length, new CompareHuffmanTrees());
    // Wypenia kolejk limi.
    for (HuffData nextSymbol : symbols) {
      BinaryTree < HuffData > aBinaryTree =
          new BinaryTree < HuffData > (nextSymbol, null, null);
      theQueue.offer(aBinaryTree);
    }

    // Buduje drzewo.
    while (theQueue.size() > 1) {
      BinaryTree < HuffData > left = theQueue.poll();
      BinaryTree < HuffData > right = theQueue.poll();
      double wl = left.getData().weight;
      double wr = right.getData().weight;
      HuffData sum = new HuffData(wl + wr, null);
      BinaryTree < HuffData > newTree =
          new BinaryTree < HuffData > (sum, left, right);
      theQueue.offer(newTree);
    }

    // Kolejka powinna teraz zawiera tylko jeden element.
    huffTree = theQueue.poll();
  }

  /** Wywietla wynikowe kody.
      @param out Obiekt PrintStream, w ktrym naley umieszcza dane.
      @param code Kod do tego wza.
      @param tree Aktualny wze drzewa.
   */
  private void printCode(PrintStream out, String code,
                         BinaryTree < HuffData > tree) {
    HuffData theData = tree.getData();
    if (theData.symbol != null) {
      if (theData.symbol.equals(" ")) {
        out.println("spacja: " + code);
      }
      else {
        out.println(theData.symbol + ": " + code);
      }
    }
    else {
      printCode(out, code + "0", tree.getLeftSubtree());
      printCode(out, code + "1", tree.getRightSubtree());
    }
  }

  /** Metoda dekodujca komunikat przekazany jako cig znakw
      '0' i '1'.
      @param codedMessage Komunikat wejciowy jako tekst ze znakami zer i jedynek.
      @return Zdekodowany komunikat jako obiekt String.
   */
  public String decode(String codedMessage) {
    StringBuilder result = new StringBuilder();
    BinaryTree < HuffData > currentTree = huffTree;
    for (int i = 0; i < codedMessage.length(); i++) {
      if (codedMessage.charAt(i) == '1') {
        currentTree = currentTree.getRightSubtree();
      }
      else {
        currentTree = currentTree.getLeftSubtree();
      }
      if (currentTree.isLeaf()) {
        HuffData theData = currentTree.getData();
        result.append(theData.symbol);
        currentTree = huffTree;
      }
    }
    return result.toString();
  }

}
