import java.util.*;

/** Przeksztaca zapis infiksowy z nawiasami na wyraenie w odwrotnej notacji polskiej.
*   @author Koffman & Wolfgang
*/

public class InfixToPostfixParens {

  // klasa zagniedona
  /** Klasa zgaszajca bd skadniowy. */
  public static class SyntaxErrorException
      extends Exception {
    /** Konstruuje wyjtek SyntaxErrorException ze wskazanym komunikatem.
        @param message Komunikat.
     */
    SyntaxErrorException(String message) {
      super(message);
    }
  }

  // pola danych
  /** Stos operatorw. */
  private Stack < Character > operatorStack;

  /** Operatory. */
  private static final String OPERATORS = "+-*/()";

  /** Priorytety operatorw, kolejno dopasowana do staej OPERATORS. */
  private static final int[] PRECEDENCE = {
      1, 1, 2, 2, -1, -1};

  /** Wyraenie w odwrotnej notacji polskiej. */
  private StringBuilder postfix;

  /** Konwertuje wyraenie z zapisu infiksowego do postaci ONP.
      @param infix Wyraenie w zapisie infiksowym.
      @throws SyntaxErrorException
   */
  public String convert(String infix) throws SyntaxErrorException {
    operatorStack = new Stack < Character > ();
    postfix = new StringBuilder();
    StringTokenizer infixTokens = new StringTokenizer(infix);
    try {
      // Przetwarza kady token wyraenia.
      while (infixTokens.hasMoreTokens()) {
        String nextToken = infixTokens.nextToken();
        char firstChar = nextToken.charAt(0);
        // Czy jest to operand?
        if (Character.isJavaIdentifierStart(firstChar)
            || Character.isDigit(firstChar)) {
          postfix.append(nextToken);
          postfix.append(' ');
        } // Czy jest to operator?
        else if (isOperator(firstChar)) {
          processOperator(firstChar);
        }
        else {
          throw new SyntaxErrorException
              ("Natrafiono na nieznany znak: "
               + firstChar);
        }
      } // Koniec ptli while.
      // Zdjcie pozostaych operatorw i dodanie ich do
      // wynikowego wyraenia.
      while (!operatorStack.empty()) {
        char op = operatorStack.pop();
        // Jeli istnieje na stosie '(', wyraenie nie ma zrwnowonych nawiasw.
        if (op == '(')
          throw new SyntaxErrorException(
              "Niedopasowany nawias otwierajcy");
        postfix.append(op);
        postfix.append(' ');
      }
      // asercja: Stos jest pusty, zwrcenie wyniku.
      return postfix.toString();
    }
    catch (EmptyStackException ex) {
      throw new SyntaxErrorException
          ("Bd skadniowy: Stos jest pusty.");
    }
  }

  /** Metoda przetwarzajca operatory.
      @param op Operator
      @throws EmptyStackException
   */
  private void processOperator(char op) {
    if (operatorStack.empty() || op == '(') {
      operatorStack.push(op);
    }
    else {
      // Pobranie elementu ze szczytu stosu
      // i zapisanie do zmiennej topOp.
      char topOp = operatorStack.peek();
      if (precedence(op) > precedence(topOp)) {
        operatorStack.push(op);
      }
      else {
        // Zdjcie ze stosu wszystkich operatorw o priorytecie
        // wyszym lub rwnym priorytetowi op.
        while (!operatorStack.empty()
               && precedence(op) <= precedence(topOp)) {
          operatorStack.pop();
          if (topOp == '(') {
            // Zdjto dopasowany '(' - wyjcie z ptli.
            break;
          }
          postfix.append(topOp);
          postfix.append(' ');
          if (!operatorStack.empty()) {
            // Ponowne ustawienie topOp.
            topOp = operatorStack.peek();
          }
        }

        // asercja: Stos operatorw jest pusty lub
        //          priorytet aktualnego operatora >
        //          priorytet operatora ze szczytu stosu.
        if (op != ')')
          operatorStack.push(op);
      }
    }
  }

  /** Okrela, czy znak jest operatorem.
      @param ch Znak do sprawdzenia.
      @return Zwraca warto true, jeli ch jest operatorem.
   */
  private boolean isOperator(char ch) {
    return OPERATORS.indexOf(ch) != -1;
  }

  /** Okrela priorytet operatora.
      @param op Operator.
      @return Zwraca priorytet operatora op.
   */
  private int precedence(char op) {
    return PRECEDENCE[OPERATORS.indexOf(op)];
  }
}

