/** Tablica mieszajca z adresacj otwart.
 *  @author Koffman and Wolfgang
 */

public class HashtableOpen < K, V >
    implements KWHashMap < K, V > {
  // pola danych
  private Entry < K, V > [] table;
  private static final int START_CAPACITY = 101;
  private double LOAD_THRESHOLD = 0.75;
  private int numKeys;
  private int numDeletes;
  private final Entry < K, V > DELETED =
      new Entry < K, V > (null, null);

  // konstruktor
  public HashtableOpen() {
    table = new Entry[START_CAPACITY];
  }

  /** Zawiera pary klucz-warto dla tablicy mieszajcej. */
  private static class Entry < K, V > {

    /** Klucz. */
    private K key;

    /** Warto. */
    private V value;

    /** Tworzy now par nazwa-klucz.
        @param key Klucz.
        @param value Warto.
     */
    public Entry(K key, V value) {
      this.key = key;
      this.value = value;
    }

    /** Pobiera klucz.
        @return Klucz.
     */
    public K getKey() {
      return key;
    }

    /** Pobiera warto.
        @return Warto.
     */
    public V getValue() {
      return value;
    }

    /** Ustawia warto.
        @param val Nowa warto.
        @return Stara warto.
     */
    public V setValue(V val) {
      V oldVal = value;
      value = val;
      return oldVal;
    }
  }

  /** Zwraca liczb wpisw w tablicy mieszajcej. */
  public int size() {
    return numKeys;
  }

  /** Zwraca true, jeli nie ma adnych elementw. */
  public boolean isEmpty() {
    return numKeys == 0;
  }

  /** Znajduje poszukiwany klucz lub pierwsze wolne miejsce w acuchu
      wyszukiwania z zastosowaniem prbkowania liniowego.
      pocz: Tablica nie jest pena.
      @param key Klucz szukanego obiektu.
      @return Pozycja poszukwanego obiektu lub gdy go nie ma,
              pozycja pierwszego wolnego miejsca.
   */
  private int find(Object key) {
    // Wyliczenie indeksu pocztkowego.
    int index = key.hashCode() % table.length;
    if (index < 0)
      index += table.length; // Niech jest to warto dodatnia.

      // Zwikszanie index a do znalezienia pustego miejsca lub
      // poszukiwanego klucza.
    while ( (table[index] != null)
           && (!key.equals(table[index].key))) {
      index++;
      // Sprawdzenie zawinicia.
      if (index >= table.length)
        index = 0; // Zawinicie.
    }
    return index;
  }

  /** Metoda get() klasy HashtableOpen.
      @param key Poszukiwany klucz.
      @return Jeli klucz odnaleziono, zwizana z nim warto.
              W przeciwnym razie warto null.
   */
  public V get(Object key) {
    // Znalezienie pierwszego, pustego elementu tablicy
    // lub elementu zawierajcego klucz.
    int index = find(key);

    // Jeli wyszukanie powiodo si, zwr warto.
    if (table[index] != null)
      return table[index].value;
    else
      return null; // Klucza nie odnaleziono.
  }

  /** Metoda put() klasy HashtableOpen.
      koc: Para klucz-warto jest wstawiona do tabeli a warto
            numKeys zinkrementowan. Jeli klucz znajduje si w tablicy
            jego warto zostaje zmieniona zgodnie z przekazanym argumentem
            Warto numKeys nie ulega wtedy zmianie. Gdy przekroczono prg
            LOAD_THRESHOLD, dochodzi do rozszerzenia tablicy.
      @param key Klucz wstawianego elementu.
      @param value Warto dla klucza.
      @return Jeli klucz istnia, jego star warto. Jeli nie istnia, warto null.
   */
  public V put(K key, V value) {
    // Znalezienie pierwszego, pustego elementu tablicy
    // lub elementu zawierajcego klucz.
    int index = find(key);

    // Jeli odnaleziono pusty element, wstaw wpis.
    if (table[index] == null) {
      table[index] = new Entry < K, V > (key, value);
      numKeys++;
      // Sprawd, czy potrzebne jest zwikszenie rozmiaru.
      double loadFactor =
          (double) (numKeys + numDeletes) / table.length;
      if (loadFactor > LOAD_THRESHOLD)
        rehash();
      return null;
    }

    // asercja: Znaleziono element tablicy zawierajcy klucz.
    // Zastp warto dla tego klucza.
    V oldVal = table[index].value;
    table[index].value = value;
    return oldVal;
  }

  /** Rozszerzenie rozmiaru, jeli wspczynnik wypenienia jest wikszy od LOAD_THRESHOLD.
      koc: Rozmiar tablicy jest podowojony i jest liczb nieparzyst.
            Kady nieusunity wpis oryginalnej tablicy zostaje wstawiony
            do nowej tablicy.
            Warto numKeys zostaje ustawiona na liczb wstawionych elementw.
            Warto numDeletes zostaje ustawiona na 0.
   */
  private void rehash() {
    // Zapamitaj referencj do oldTable.
    Entry < K, V > [] oldTable = table;
    // Utwrz dwa razy wiksz tablic.
    table = new Entry[2 * oldTable.length + 1];

    // Wstaw wszystkie elementy z oldTable do nowej tablicy.
    numKeys = 0;
    numDeletes = 0;
    for (int i = 0; i < oldTable.length; i++) {
      if ( (oldTable[i] != null) && (oldTable[i] != DELETED)) {
        // Wstaw wpis.
        put(oldTable[i].key, oldTable[i].value);
      }
    }
  }

/**** WICZENIE ****/
}
