import java.io.*;

/** Klasa reprezentujca cig bitw. Klasa przypomina klasy
*    String i StringBuilder, ale zawiera cig bitw 0 i 1.
*   Obiekty BitSrings maj zmienn dugo.
*   @author Koffman and Wolfgang
*/

public class BitString
    implements Cloneable, Serializable {

  /** Tablica bajtw przechowujca dane. */
  private byte[] theData;

  /** Aktualna pojemno (rozmiar tablicy bajtw) */
  private int capacity;

  /** Liczba zapamitanych bitw. */
  private int size;

  /** Tworzy pusty obiekt BitString o podanej pojemnoci pocztkowej.
      @param capacity Pojemno pocztkowa.
   */
  public BitString(int capacity) {
    theData = new byte[capacity];
    size = 0;
    this.capacity = capacity;
  }

  /** Tworzy pusty obiekt BitString o pocztkowej pojemnoci rwnej 1
   */
  public BitString() {
    this(1);
  }

  /** Tworzy mask dla wybranego bitu.
      @param i Indeks wybranego bitu.
      @return Maska z bitem ustawionym na 1 w odpowiednim miejscu.
      appropriate bit
   */
  private int mask(int i) {
    int r = i % 8;
    return 1 << r;
  }

  /** Okrela indeks bajtu zawierajcego wskazany bit.
      @param i Indeks wybieranego bitu.
      @return Zwraca indeks bajtu zawierajcego wskazany bit.
   */
  private int index(int i) {
    return i / 8;
  }

  /** Pobiera wskazany bit.
      @param i Indeks wybieranego bitu.
      @return Zwraca warto true, jeli bit zawiera warto 1.
   */
  public boolean get(int i) {
    if (i < 0 || i >= size) {
      throw new IndexOutOfBoundsException();
    }
    return (theData[index(i)] & mask(i)) != 0;
  }

  /** Ustawia wskazany bit na 1.
      @param i Indeks wybieranego bitu.
      @throws Zgasza wyjtek IndexOutOfBoundsException, jeli indeks mniejszy od 0.
   */
  public void set(int i) {
    if (i < 0) {
      throw new IndexOutOfBoundsException();
    }
    if (i >= size) {
      if (i / 8 >= capacity) {
        ensureCapacity(i / 8);
      }
      size = i + 1;
    }
    theData[index(i)] |= mask(i);
  }

  /** Ustawia wskazany bit na 0.
      @param i Indeks wybieranego bitu.
      @throws Zgasza wyjtek IndexOutOfBoundsException, jeli indeks mniejszy od 0.
   */
  public void clear(int i) {
    if (i < 0) {
      throw new IndexOutOfBoundsException();
    }
    if (i >= size) {
      if (i / 8 >= capacity) {
        ensureCapacity(i / 8);
      }
      size = i + 1;
    }
    theData[index(i)] &= ~mask(i);
  }

  /** Zapewnia, i pojemno jest co najmniej tak dua, jak
      poszukiwana warto. Jeli warto jest wiksza od aktualnej
      pojemnoci, ale mniejsza od dwukrotnoci aktualnej pojemnoci,
      dochodzi do podwojemnia aktualnej pojemnoci.
      capacity is doubled
      @param c Podana pojemno.
   */
  private void ensureCapacity(int c) {
    int newCapacity = Math.max(c, Math.max(2 * capacity, 1));
    byte[] newData = new byte[newCapacity];
    System.arraycopy(theData, 0, newData, 0, capacity);
    theData = newData;
    capacity = newCapacity;
  }

  /** Przytnij pojemno do minimalnego rozmiaru potrzebnego
      do przechowania cigu bitw o aktualnym rozmiarze.
   */
  public void trimCapacity() {
    int newCapacity = Math.max( (size + 7) / 8, 1);
    byte[] newData = new byte[newCapacity];
    System.arraycopy(theData, 0, newData, 0, newCapacity);
    theData = newData;
    capacity = newCapacity;
  }

  /** Dodaje warto na kocu cigu bitw.
      @param newBit Warto true oznacza wpisanie 1 a false wpisanie 0.
   */
  public BitString append(boolean newBit) {
    if (newBit) {
      set(size);
    }
    else {
      clear(size);
    }
    return this;
  }

  /** Metoda dodajca jeden obiekt BitString do drugiego.
      @param left BitString.
      @param right Prawy BitString.
      koc: Lewy BitString zostaje powikszony o przekazany cig bitw.
   */
  public BitString append(BitString right) {
    for (int i = 0; i < right.size(); i++) {
      append(right.get(i));
    }
    return this;
  }

  /** Tworzy kopi obiektu BitString.
      @return Kopia obiektu BitString.
   */
  public Object clone() {
    try {
      BitString theClone = (BitString)super.clone();
      int newCapacity = Math.max( (size + 7) / 8, 1);
      theClone.theData = new byte[newCapacity];
      System.arraycopy(theData, 0, theClone.theData, 0, newCapacity);
      theClone.capacity = newCapacity;
      return theClone;
    }
    catch (CloneNotSupportedException ex) {
      throw new InternalError();
    }
  }

  /** Zwraca tekst reprezentujcy obiekt BitString,
      w ktrym 1 to '1' a kade 0 to '0'.
      @return Tekstowa reprezentacja obiektu BitString.
   */
  public String toString() {
    StringBuilder result = new StringBuilder();
    for (int i = 0; i < size; i++) {
      if (get(i)) {
        result.append("1");
      }
      else {
        result.append("0");
      }
    }
    return result.toString();
  }

  /** Sprawdza, czy dwa obiekty bitStrings s sobie rwne.
      @param other Drugi obiekt BitString.
      @return Zwraca warto tru, jeli oba obiekty BitString zawieraj to samo.
   */
  public boolean equals(Object other) {
    if (other instanceof BitString) {
      BitString otherBitString =
          (BitString) other;
      for (int i = 0; i < capacity; i++) {
        if (theData[i] != otherBitString.theData[i]) {
          return false;
        }
      }
      return true;
    }
    else {
      return false;
    }
  }

  /** Metoda dodajca jeden obiekt BitString do drugiego.
      @param left BitString.
      @param right Prawy BitString
      koc: Lewy obiekt BitString zostaje uzupeniony o prawy BitString.
      result
   */
  public static void append(BitString left, BitString right) {
    left.append(right);
  }

  /** Zwraca skrt dla obiektu BitString.
      @return Skrt obiektu.
   */
  public int hashCode() {
    int result = 0;
    for (int i = 0; i < capacity; i++) {
      result = (result << 1) | (result >> 31);
      result ^= theData[i];
    }
    return result;
  }

  /** Zwraca rozmiar obiektu BitString.
      @return Rozmiar obiektu.
   */
  public int size() {
    return size;
  }
}
