import java.util.*;

/** Implementuje interfejs Queue wykorzystujc kolejk cykliczn.
 *  @author Koffman & Wolfgang
 * */

public class ArrayQueue < E >
    extends AbstractQueue < E >
    implements Queue < E > {

  // pola danych
  /** Indeks pocztku kolejki. */
  private int front;

  /** Indeks koca kolejki. */
  private int rear;

  /** Aktualny rozmiar kolejki. */
  private int size;

  /** Pojemno kolejki. */
  private int capacity;

  /** Domylna pojemno kolejki. */
  private static final int DEFAULT_CAPACITY = 10;

  /** Tablica przechowujca dane. */
  private E[] theData;

  // konstruktory
  /** Tworzy kolejk o domylnej pojemnoci pocztkowej.
   */
  public ArrayQueue() {
    this(DEFAULT_CAPACITY);
  }

  /** Tworzy kolejk o podanej pojemnoci domylnej.
      @param initCapacity Pojemno pocztkowa.
   */
  public ArrayQueue(int initCapacity) {
    capacity = initCapacity;
    theData = (E[])new Object[capacity];
    front = 0;
    rear = capacity - 1;
    size = 0;
  }

  // metody publiczne
  /** Wstawia element na kocu kolejki.
      koc: Element zosta dodany na kocu kolejki.
      @param item Element do dodania.
      @return Warto true (wstawienie zawsze jest udane).
   */
  public boolean offer(E item) {
    if (size == capacity) {
      reallocate();
    }
    size++;
    rear = (rear + 1) % capacity;
    theData[rear] = item;
    return true;
  }

  /** Zwraca element z pocztku kolejki bez jego usuwania.
      @return Element z pocztku w razie sukcesu, warto null w przypadku przeciwnym.
   */
  public E peek() {
    if (size == 0)
      return null;
    else
      return theData[front];
  }

  /** Usunicie elementu z pocztku kolejki i zwrcenie go, jeli
      kolejka nie jest pusta.
      koc: Pole front zawiera referencj do elementu, ktry wczeniej by drugi.
      @return Usuwany element w razie sukcesu, warto null w przypadku przeciwnym.
   */
  public E poll() {
    if (size == 0) {
      return null;
    }
    E result = theData[front];
    front = (front + 1) % capacity;
    size--;
    return result;
  }

/**** WICZENIE ****/

  // metody prywatne
  /** Podwojenie pojemnoci i realokacja danych.
      pocz: Tablica jest wypeniona do swej maksymalnej pojmnoci.
      koc: Pojemno zostaje podwojona i pierwsza poowa nowej
            tablicy jest wypeniona starymi danymi.
   */
  private void reallocate() {
    int newCapacity = 2 * capacity;
    E[] newData = (E[])new Object[newCapacity];
    int j = front;
    for (int i = 0; i < size; i++) {
      newData[i] = theData[j];
      j = (j + 1) % capacity;
    }
    front = 0;
    rear = size - 1;
    capacity = newCapacity;
    theData = newData;
  }

  /** Klasa wewntrzna implementujca interfejs Iterator<E>. */
  private class Iter
      implements Iterator < E > {
    // pola danych
    // Indeks kolejnego elementu. */
   private int index;

    // Zliczanie udostpnionych do tej pory elementw. */
   private int count = 0;

    // metody
    // konstruktor
    /** Inicjalizuje obiekt Iter, by wskazywa na pierwszy element kolejki.
     */
    public Iter() {
      index = front;
    }

    /** Zwraca warto true, jeli kolejka zawiera jeszcze elementy do pobrania. */
    public boolean hasNext() {
      return count < size;
    }

    /** Zwraca nastpny element kolejki.
        pocz: Indeks wskazuje na nastpny element do pobrania.
        koc: Pola index i count zostaj zinkrementowane.
        @return Zwraca element o indeksie index.
     */
    public E next() {
      if (!hasNext()) {
        throw new NoSuchElementException();
      }
      E returnValue = theData[index];
      index = (index + 1) % capacity;
      count++;
      return returnValue;
    }

    /** Usuwa aktualny element wskazywany przez obiekt Iter  nie zaimplementowano. */
    public void remove() {
      throw new UnsupportedOperationException();
    }
  }

}
