/*---------------------------------------------------------------------------*/
    #include <errno.h>
    #include <fcntl.h>
    #include <stdarg.h>
    #include <stdio.h>
    #include <stdlib.h>
    #include <sys/stat.h>
    #include <sys/types.h>
    #include <unistd.h>

    #include <bfd.h>
    #include <dis-asm.h>


    /* rodzaje operandw */
    enum op_type { op_unk, op_reg, op_imm, op_expr, op_bptr, op_dptr, 
               op_wptr };

    struct ASM_INSN {
        char mnemonic[16];
        char src[32];
        char dest[32];
        char arg[32];
        enum op_type src_type, dest_type, arg_type;
    } curr_insn;


    enum op_type optype( char *op){
        if ( op[0] == '%' ) { return(op_reg); }
        if ( op[0] == '$' ) { return(op_imm); }
        if ( strchr(op, '(')) { return(op_expr); }
        if ( strncmp( op, "BYTE PTR", 8)) { return(op_bptr); }
        if ( strncmp( op, "DWORD PTR", 9)) { return(op_dptr); }
        if ( strncmp( op, "WORD PTR", 8)) { return(op_wptr); }
        return(op_unk);
    }

    /* moemy sobie na to pozwoli, gdy Intel ma niewiele instrukcji typu 'j' */
    #define JCC_INSN 'j'
    #define JMP_INSN "jmp"
    #define LJMP_INSN "ljmp"
    #define CALL_INSN "call"
    #define RET_INSN "ret"
    
    enum flow_type { flow_branch, flow_cond_branch, flow_call, flow_ret, 
                 flow_none };

    enum flow_type insn_flow_type( char *insn ) {
        if (! strncmp( JMP_INSN, insn, 3)  ||
            ! strncmp( LJMP_INSN, insn, 4) ) {
            return( flow_branch );
        } else if ( insn[0] == JCC_INSN ) {
            return( flow_cond_branch ) ;
        } else if ( ! strncmp( CALL_INSN, insn, 4) ) {
            return( flow_call );
        } else if ( ! strncmp( RET_INSN, insn, 3) ) {
            return( flow_ret );
        }
        return( flow_none );
    }

    int disprintf(FILE *stream, const char *format, ...){
        va_list args;
        char *str;

        va_start (args, format);
        str = va_arg(args, char*);

        /* fatalnie, libopcodes przekazuje w wywoaniu one mnemonik lub operand
           i przekazuje src dwukrotnie */
        if ( ! curr_insn.mnemonic[0] ) {
            strncpy(curr_insn.mnemonic, str, 15);
        } else if ( ! curr_insn.src[0] ) {
            strncpy(curr_insn.src, str, 31);
            curr_insn.src_type = optype(curr_insn.src);
        } else if ( ! curr_insn.dest[0] ) {
            strncpy(curr_insn.dest, str, 31);
            curr_insn.dest_type = optype(curr_insn.dest);
            if (strncmp(curr_insn.dest, "DN", 2) == 0)
                    curr_insn.dest[0] = 0;
        } else {
            if ( ! strcmp( curr_insn.src, curr_insn.dest ) ) {
                /* src zosta przekazany dwukrotnie */
                strncpy(curr_insn.dest, str, 31);
                curr_insn.dest_type = optype(curr_insn.dest);
            } else {
                strncpy(curr_insn.arg, str, 31);
                curr_insn.arg_type = optype(curr_insn.arg);
            }
        }
        va_end (args);
    
        return(0);
    }

    void print_insn( void ) {
        printf("\t%s", curr_insn.mnemonic);
        if ( curr_insn.src[0] ) {
            printf("\t%s", curr_insn.src );
            if ( curr_insn.dest[0] ) {
                printf(", %s", curr_insn.dest );
                if ( curr_insn.arg[0] ) {
                    printf(", %s", curr_insn.arg );
                }
            }
        }
        return;
    }

    int rva_from_op( char *op, unsigned long *rva ) {
        if ( *op == '*' )     return(0);    /* wskanik */
        if ( *op == '$' )     op++;
        if ( isxdigit(*op) ) {
            *rva = strtoul(curr_insn.src, NULL, 16);
            return(1);
        }
        return(0);
    }

    static void disassemble( bfd *b, unsigned long rva, int nest );

    int disassemble_forward( disassembler_ftype disassemble_fn, 
                disassemble_info *info, unsigned long rva, int nest ) {
        int i, good_rva, size, offset, bytes = 0;
        unsigned long branch_rva;
        enum flow_type flow;

        if (! nest )     
            return(0);
    
        /* pobranie pooenia rva */
        offset = rva - info->buffer_vma;
        
        /* zabezpieczeni przed nieskoczonymi ptlami */
        nest--;

        while ( bytes < info->buffer_length ) {
            /* to trzeba zweryfikowa z powodu rozgazienia kodu */
            if ( rva < info->buffer_vma || 
                      rva >= info->buffer_vma + info->buffer_length ) {
                /* rekursywnie disassemble( ), a nastpnie wyjcie */
                disassemble( NULL, rva + bytes, nest );
                return(0);
            }

            /* wywoanie deasemblera libopcodes */
            memset( &curr_insn, 0, sizeof( struct ASM_INSN ));
            size = (*disassemble_fn)(rva + bytes, info); 

            /* -- analiza zdeasemblowanych instrukcji -- */
    
            /* -- wywietla nazwy symboli jako etykiety -- */
            printf("%8X:   ", rva + bytes);
            /* wywietla bajty szesnastkowo */
            for ( i = 0; i < 8; i++ ) {
                if ( i < size )     
                    printf("%02X ", info->buffer[offset+bytes+i]);
                else 
                    printf("   ");
            }
            print_insn( );
            printf("\n");
            bytes += size;    /* przeniesienie pozycji w buforze */
            /* sprawdzenie rodzaju instrukcji */
            flow = insn_flow_type( curr_insn.mnemonic ); 
            if ( flow == flow_branch || flow == flow_cond_branch || 
                 flow == flow_call ) {
                /* pjcie rozgazieniem programu */
                good_rva = 0;
                if ( curr_insn.src_type == op_bptr ||
                     curr_insn.src_type == op_wptr ||
                     curr_insn.src_type == op_dptr    ) {
                    good_rva = rva_from_op( curr_insn.src, 
                                &branch_rva );
                }
                if ( good_rva ) {
                    printf(";------------------ ROZGAZIENIE %X\n",
                            branch_rva );
                    disassemble_forward( disassemble_fn, info, 
                                branch_rva, nest );
                }
            }
            if ( flow == flow_branch || flow == flow_ret ) {
                /* koniec wykonywania: opuszczenie ptli */
                bytes = info->buffer_length;
                printf(";------------------------- KONIEC\n");
                continue;
            }
        }
        return( bytes );
    }

    struct RVA_SEC_INFO {
        unsigned long rva;
        asection *section;
    };

    static void find_rva( bfd *b, asection *section, PTR data ){
        struct RVA_SEC_INFO *rva_info = data;
        if ( rva_info->rva >= section->vma && 
             rva_info->rva < section->vma + bfd_section_size( b, section ))
            /* mamy zwycizc */
            rva_info->section = section;
        return;
    }
         
    static void disassemble( bfd *b, unsigned long rva, int nest ) {
        static disassembler_ftype disassemble_fn;
        static disassemble_info info = {0};
        static bfd *bfd = NULL;
        struct RVA_SEC_INFO rva_info;
        unsigned char *buf;
        int size;

        if ( ! bfd ) {
            if ( ! b )     return;
            bfd = b;
            /* inicjalizacja wszystkiego */
            INIT_DISASSEMBLE_INFO(info, stdout, disprintf);
            info.arch = bfd_get_arch(b);
            info.mach = bfd_mach_i386_i386;
            info.flavour = bfd_get_flavour(b);
            info.endian = b->xvec->byteorder;
            info.display_endian = BFD_ENDIAN_LITTLE;
            disassemble_fn = print_insn_i386_att;
        }
    
        /* szukanie sekcji zawierajcych rva */
        rva_info.rva = rva;
        rva_info.section = NULL;
        bfd_map_over_sections( bfd, find_rva, &rva_info );
        if ( ! rva_info.section ) 
            return;
        size = bfd_section_size( bfd, rva_info.section );
        buf = calloc( size, 1 );
        /* bdziemy okrutni i zwolnimy calloc przy exit( ) */
        if ( ! bfd_get_section_contents( bfd, rva_info.section, buf, 0, 
                                            size ) )
            return;
    
        info.section = rva_info.section;            /* sekcja do deasemblacji */
        info.buffer = buf;                          /* bufor bajtw do deasemblacji */
        info.buffer_length = size;                  /* rozmiar bufora */
        info.buffer_vma = rva_info.section->vma;    /* podstawa RVA bufora */
        
        disassemble_forward( disassemble_fn, &info, rva, nest );
    
        return;
    }
    
    static void print_section_header( asection *s, const char *mode ) {
    
        printf("Deasemblacja sekcji %s jako %s\n", s->name, mode );
        printf("RVA: %08X LMA: %08X Znaczniki: %08X Rozmiar: %X\n", s->vma,
                s->lma, s->flags, s->_cooked_size );
        printf( "--------------------------------------------------------"
              "------------------------\n" );
        return;
    }

    static void disasm_section( bfd *b, asection *section, PTR data ){
        int i, j, size;
        unsigned char *buf;

        /* interesuj nas tylko sekcje danych */
        if ( ! section->flags & SEC_ALLOC )  return;
        if ( ! section->flags & SEC_LOAD ) return;
        if ( section->flags & SEC_LINKER_CREATED ) return;
        if ( section->flags & SEC_CODE) {
            return;
        } else if ( (section->flags & SEC_DATA ||
                 section->flags & SEC_READONLY) &&
                section->flags & SEC_HAS_CONTENTS  ) {
            /* wywietlenie zawartoci danych sekcji */
            size = bfd_section_size( b, section );
            buf = calloc( size, 1 );
            if ( ! bfd_get_section_contents( b, section, buf, 0, size ) )
                return;
            print_section_header( section, "data" );
            for ( i = 0; i < size; i +=16 ) {
                printf( "%08X:   ", section->vma + i );
                for (j = 0; j < 16 && j+i < size; j++ )  
                    printf("%02X ", buf[i+j] );
                for ( ; j < 16; j++ )
                    printf("   ");
                printf("  ");
                for (j = 0; j < 16 && j+i < size; j++ ) 
                    printf("%c", isprint(buf[i+j]) ? buf[i+j] : '.');
                printf("\n");
            }
            printf("\n\n");
            free( buf );
        }
        return;
    }

    int main( int argc, char **argv ) {
          struct stat s;
        bfd *infile;

          if ( argc < 2 ) {
                fprintf(stderr, "Uycie: %s nazwa_pliku\n", argv[0]);
                return(1);
          }
              if ( stat( argv[1], &s) ) {
                    fprintf(stderr, "Bd: %s\n", strerror(errno) );
                    return(2);
              }

        bfd_init( );

        /* otwarcie pliku wejciowego do odczytu */
        infile = bfd_openr( argv[1], NULL );
        if ( ! infile ) {
            bfd_perror( " Bd w pliku wejciowym" );
            return(3);
        }

        if ( bfd_check_format (infile, bfd_object ) || 
              bfd_check_format(infile, bfd_archive ) ) {
            /* deasemblacja w przd od punktu wejcia */
            disassemble( infile, bfd_get_start_address(infile), 10 );
            /* deasemblacja sekcji danych */
            bfd_map_over_sections( infile, disasm_section, NULL );
        } else  {
            fprintf( stderr, " Bd: nieznany format pliku\n");
            return(5);
        }

        bfd_close(infile);
    
          return(0);
    }
/*-----------------------------------------------------------------------*/
