---------------------------------------------------------------------------
-- SQL w praktyce. Jak dzięki danym uzyskiwać cenne informacje. Wydanie II
-- Anthony DeBarros

-- Rozdział 15. Przykłady z kodem
----------------------------------------------------------------------------


-- Listing 15.1: Ładowanie rozszerzenia PostGIS

CREATE EXTENSION postgis;

SELECT postgis_full_version(); -- wyświetla wersję rozszerzenia PostGIS

-- Listing 15.2: Uzyskiwanie danych formatu WKT dla identyfikatora SRID 4326

SELECT srtext
FROM spatial_ref_sys
WHERE srid = 4326;

-- Listing 15.3: Zastosowanie ST_GeomFromText() do tworzenia obiektów przestrzennych

SELECT ST_GeomFromText('POINT(-74.9233606 42.699992)', 4326);

SELECT ST_GeomFromText('LINESTRING(-74.9 42.7, -75.1 42.7)', 4326);

SELECT ST_GeomFromText('POLYGON((-74.9 42.7, -75.1 42.7,
                                 -75.1 42.6, -74.9 42.7))', 4326);

SELECT ST_GeomFromText('MULTIPOINT (-74.9 42.7, -75.1 42.7)', 4326);

SELECT ST_GeomFromText('MULTILINESTRING((-76.27 43.1, -76.06 43.08),
                                        (-76.2 43.3, -76.2 43.4,
                                         -76.4 43.1))', 4326);

SELECT ST_GeomFromText('MULTIPOLYGON((
                                     (-74.92 42.7, -75.06 42.71,
                                      -75.07 42.64, -74.92 42.7),
                                     (-75.0 42.66, -75.0 42.64,
                                      -74.98 42.64, -74.98 42.66,
                                      -75.0 42.66)))', 4326);

-- Listing 15.4: Użycie funkcji ST_GeogFromText() do tworzenia obiektów przestrzennych

SELECT
ST_GeogFromText('SRID=4326;MULTIPOINT(-74.9 42.7, -75.1 42.7, -74.924 42.6)');

-- Listing 15.5: Funkcje powiązane z tworzeniem punktów

SELECT ST_PointFromText('POINT(-74.9233606 42.699992)', 4326);

SELECT ST_MakePoint(-74.9233606, 42.699992);
SELECT ST_SetSRID(ST_MakePoint(-74.9233606, 42.699992), 4326);

-- Listing 15.6: Funkcje przeznaczone do tworzenia obiektów LineString

SELECT ST_LineFromText('LINESTRING(-105.90 35.67,-105.91 35.67)', 4326);
SELECT ST_MakeLine(ST_MakePoint(-74.9, 42.7), ST_MakePoint(-74.1, 42.4));

-- Listing 15.7: Funkcje przewidziane do tworzenia wielokątów

SELECT ST_PolygonFromText('POLYGON((-74.9 42.7, -75.1 42.7,
                                    -75.1 42.6, -74.9 42.7))', 4326);

SELECT ST_MakePolygon(
           ST_GeomFromText('LINESTRING(-74.92 42.7, -75.06 42.71,
                                       -75.07 42.64, -74.92 42.7)', 4326));

SELECT ST_MPolyFromText('MULTIPOLYGON((
                                       (-74.92 42.7, -75.06 42.71,
                                        -75.07 42.64, -74.92 42.7),
                                       (-75.0 42.66, -75.0 42.64,
                                        -74.98 42.64, -74.98 42.66,
                                        -75.0 42.66)
                                      ))', 4326);


-- ANALIZOWANIE DANYCH DOTYCZĄCYCH TARGÓW ROLNICZYCH
-- https://www.ams.usda.gov/local-food-directories/farmersmarkets


-- Listing 15.8: Tworzenie i ładowanie tabeli farmers_markets

CREATE TABLE farmers_markets (
    fmid bigint PRIMARY KEY,
    market_name text NOT NULL,
    street text,
    city text,
    county text,
    st text NOT NULL,
    zip text,
    longitude numeric(10,7),
    latitude numeric(10,7),
    organic text NOT NULL
);

COPY farmers_markets
FROM 'C:\TwojKatalog\farmers_markets.csv'
WITH (FORMAT CSV, HEADER);

SELECT count(*) FROM farmers_markets; -- should return 8,681 rows

-- Listing 15.9: Tworzenie i indeksowanie kolumny typu geography
-- Dostępna jest też funkcja: https://postgis.net/docs/AddGeometryColumn.html

-- Dodanie kolumny
ALTER TABLE farmers_markets ADD COLUMN geog_point geography(POINT,4326);

-- Wypełnij teraz tę kolumnę wartościami szerokości i długości geograficznej
UPDATE farmers_markets
SET geog_point = 
     ST_SetSRID(
               ST_MakePoint(longitude,latitude)::geography,4326
               );

-- Dodaj indeks danych przestrzennych (R-drzewo) za pomocą interfejsu GIST
CREATE INDEX market_pts_idx ON farmers_markets USING GIST (geog_point);

-- Wyświetlenie kolumny z danymi geograficznymi
SELECT longitude,
       latitude,
       geog_point,
       ST_AsEWKT(geog_point)
FROM farmers_markets
WHERE longitude IS NOT NULL
LIMIT 5;

-- Listing 15.10: Użycie funkcji ST_DWithin() do zlokalizowania tragów rolniczych oddalonych nie więcej niż 10 kilometrów od danego punktu

SELECT market_name,
       city,
       st,
       geog_point
FROM farmers_markets
WHERE ST_DWithin(geog_point,
                 ST_GeogFromText('POINT(-93.6204386 41.5853202)'),
                 10000)
ORDER BY market_name;

-- Listing 15.11: Zastosowanie funkcji ST_Distance() do obliczenia odległości między stadionami Yankee Stadium i Citi Field
-- 1609.344 metry/mila

SELECT ST_Distance(
                   ST_GeogFromText('POINT(-73.9283685 40.8296466)'),
                   ST_GeogFromText('POINT(-73.8480153 40.7570917)')
                   ) / 1609.344 AS mets_to_yanks;

-- Listing 15.12: Użycie funkcji ST_Distance() dla każdego wiersza tabeli farmers_markets

SELECT market_name,
       city,
       round(
           (ST_Distance(geog_point,
                        ST_GeogFromText('POINT(-93.6204386 41.5853202)')
                        ) / 1609.344)::numeric, 2
            ) AS miles_from_dt
FROM farmers_markets
WHERE ST_DWithin(geog_point,
                 ST_GeogFromText('POINT(-93.6204386 41.5853202)'),
                 10000)
ORDER BY miles_from_dt ASC;

-- Listing 15.13: Użycie operatora odległości <-> w ramach wyszukiwania najbliższych sąsiadów

SELECT market_name,
       city,
       st,
       round(
           (ST_Distance(geog_point,
                        ST_GeogFromText('POINT(-68.2041607 44.3876414)')
                        ) / 1609.344)::numeric, 2
            ) AS miles_from_bh
FROM farmers_markets
ORDER BY geog_point <-> ST_GeogFromText('POINT(-68.2041607 44.3876414)')
LIMIT 3;

-- Praca z plikami shapefile

-- Zasoby:
   -- Pliki shapefile TIGER/Line® i pliki TIGER/Line®
   -- https://www.census.gov/geo/maps-data/data/tiger-line.html
   -- Kartograficzne pliki shapefile granic hrabstw
   -- https://www.census.gov/geo/maps-data/data/cbf/cbf_counties.html


-- Importowanie (do zastosowania w wierszu poleceń, jeśli używa się systemu macOS lub Linux; zajrzyj do rozdziału 18.)
shp2pgsql -I -s 4269 -W LATIN1 tl_2019_us_county.shp us_counties_2019_shp | psql -d analysis -U postgres

-- Listing 15.14: Sprawdzanie reprezentacji WKT wartości kolumny geom

SELECT ST_AsText(geom)
FROM us_counties_2019_shp
ORDER BY gid
LIMIT 1;

-- Listing 15.15: Znajdowanie hrabstw o największej powierzchni za pomocą funkcji ST_Area()

SELECT name,
       statefp AS st,
       round(
             ( ST_Area(geom::geography) / 2589988.110336 )::numeric, 2
            )  AS square_miles
FROM us_counties_2019_shp
ORDER BY square_miles DESC
LIMIT 5;

-- Listing 15.16: Użycie funkcji ST_Within() do znalezienia hrabstwa powiązanego z daną parą współrzędnych

SELECT sh.name,
       c.state_name
FROM us_counties_2019_shp sh JOIN us_counties_pop_est_2019 c
    ON sh.statefp = c.state_fips AND sh.countyfp = c.county_fips
WHERE ST_Within(
         'SRID=4269;POINT(-118.3419063 34.0977076)'::geometry, geom
);

-- Listing 15.17: Zastosowanie funkcji ST_DWithin() do ustalenia liczby ludzi mieszkających w pobliżu miast Lincoln w stanie Nebraska

SELECT sum(c.pop_est_2019) AS pop_est_2019
FROM us_counties_2019_shp sh JOIN us_counties_pop_est_2019 c
    ON sh.statefp = c.state_fips AND sh.countyfp = c.county_fips
WHERE ST_DWithin(sh.geom::geography, 
          ST_GeogFromText('SRID=4269;POINT(-96.699656 40.811567)'),
          80467);
          
-- Uwaga: możesz przyspieszyć powyższe zapytanie przez utworzenie indeksu funkcyjnego,
-- który obejmuje rzutowanie typu kolumny geom na typ kolumny geography.
CREATE INDEX us_counties_2019_shp_geog_idx ON us_counties_2019_shp USING GIST (CAST(geom AS geography));

-- Listing 15.18: Wyświetlanie hrabstw w pobliżu Lincoln w stanie Nebraska

SELECT sh.name,
       c.state_name,
       c.pop_est_2019,
       ST_Transform(sh.geom, 4326) AS geom
FROM us_counties_2019_shp sh JOIN us_counties_pop_est_2019 c
    ON sh.statefp = c.state_fips AND sh.countyfp = c.county_fips
WHERE ST_DWithin(sh.geom::geography, 
          ST_GeogFromText('SRID=4269;POINT(-96.699656 40.811567)'),
          80467);


-- Złączenia danych przestrzennych
-- Danych dotyczące dróg gruntowych i wodnych w hrabstwie Santa Fe
-- http://www.santafenm.gov/santa_fe_river

-- Zestaw plików shapefile TIGER/Line dotyczących dróg gruntowych i wodnych dla spisu ludności z roku 2019 
-- https://www.census.gov/geo/maps-data/data/tiger-line.html
-- https://www.census.gov/cgi-bin/geo/shapefiles/index.php?year=2019&layergroup=Roads
-- https://www.census.gov/cgi-bin/geo/shapefiles/index.php?year=2019&layergroup=Water

-- RTTYP - Opis kodów typów dróg
-- https://www.census.gov/geo/reference/rttyp.html
-- C Hrabstwo
-- I Międzystanowa
-- M Ogólna nazwa
-- O Inne
-- S Stanowe
-- U Stany Zjednoczone

-- Kod klas opcji MTFCC MAF/TIGER
-- https://www.census.gov/geo/reference/mtfcc.html
-- Tutaj H3010: naturalne cieki wodne


-- Importowanie (do zastosowania w wierszu poleceń, jeśli używa się systemu macOS lub Linux; zajrzyj do rozdziału 18.)
shp2pgsql -I -s 4269 -W LATIN1 tl_2019_35049_linearwater.shp santafe_linearwater_2019 | psql -d analysis -U postgres
shp2pgsql -I -s 4269 -W LATIN1 tl_2019_35049_roads.shp santafe_roads_2019 | psql -d analysis -U postgres

-- Listing 15.19: Użycie funkcji ST_GeometryType() do określenia elementu geometrycznego

SELECT ST_GeometryType(geom)
FROM santafe_linearwater_2019
LIMIT 1;

SELECT ST_GeometryType(geom)
FROM santafe_roads_2019
LIMIT 1;

-- Listing 15.20: Złączenie danych przestrzennych za pomocą funkcji ST_Intersects() w celu znalezienia dróg przecinających się z rzeką Santa Fe

SELECT water.fullname AS waterway,
       roads.rttyp,
       roads.fullname AS road
FROM santafe_linearwater_2019 water JOIN santafe_roads_2019 roads
    ON ST_Intersects(water.geom, roads.geom)
WHERE water.fullname = 'Santa Fe Riv' 
      AND roads.fullname IS NOT NULL
ORDER BY roads.fullname;

-- Listing 15.21: Zastosowanie funkcji ST_Intersection() do ustalenia miejsc, w których drogi przecinają rzekę

SELECT water.fullname AS waterway,
       roads.rttyp,
       roads.fullname AS road,
       ST_AsText(ST_Intersection(water.geom, roads.geom))
FROM santafe_linearwater_2019 water JOIN santafe_roads_2019 roads
    ON ST_Intersects(water.geom, roads.geom)
WHERE water.fullname = 'Santa Fe Riv'
      AND roads.fullname IS NOT NULL
ORDER BY roads.fullname;

