// Wersja końcowa 
package com.apress.springrecipes.court.web;

import com.apress.springrecipes.court.domain.Member;
import com.apress.springrecipes.court.service.MemberService;

import org.springframework.beans.factory.annotation.Autowired;

import org.springframework.stereotype.Controller;

import org.springframework.ui.Model;

import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;

import org.springframework.web.bind.annotation.*;

import java.util.Set;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.ValidatorFactory;


@Controller
// Wiązanie kontrolera z wszystkimi adresami URL z członem /member/* 
// Początkowy widok jest ustalany na podstawie nazwy zwracanej przez domyślną metodę obsługi żądań GET 
@RequestMapping("/member/*")
@SessionAttributes("guests")
public class MemberController {
    // Walidator JSR-303
    private static Validator validator;
    private MemberService memberService;

    // Podłączanie usługi i walidatora w konstruktorze
    @Autowired
    public MemberController(MemberService memberService) {
        this.memberService = memberService;

        // Inicjowanie sprawdzania poprawności zgodnie ze specyfikacją JSR-303
        ValidatorFactory validatorFactory = Validation.buildDefaultValidatorFactory();
        validator = validatorFactory.getValidator();
    }

    // Kontroler zawsze szuka najpierw domyślnej metody obsługi żądań GET (nazwa nie ma tu znaczenia).
    // Tu ta metoda nosi nazwę setupForm, co ułatwia jej identyfikację
    @RequestMapping(method = RequestMethod.GET)
    // Metoda przyjmuje parametr typu Model używany do ustawienia obiektu typu Member
    public String setupForm(Model model) {
        model.addAttribute("member", new Member());
        model.addAttribute("guests", memberService.list());

        // Zwracanie widoku memberList. Jest on wiązany z plikiem
        // /WEB-INF/jsp/memberList.jsp
        return "memberList";
    }

	// Kontroler zawsze szuka domyślnej metody obsługi żądań POST (nazwa nie ma tu znaczenia),
	// gdy zgłaszane jest żądanie danego adresu URL (@RequestMapping(/member/*)).
	// Tu ta metoda nosi nazwę submitForm, co ułatwia jej identyfikację
    @RequestMapping(method = RequestMethod.POST)
    public String submitForm(@ModelAttribute("member")
    Member member, BindingResult result, Model model) {
        // Wystąpiły błędy JSR-303 (sprawdzanie poprawności ilustruje klasa Member) 
        Set<ConstraintViolation<Member>> violations = validator.validate(member);

        // Przejście w pętli po możliwych błędach
        for (ConstraintViolation<Member> violation : violations) {
            String propertyPath = violation.getPropertyPath().toString();
            String message = violation.getMessage();
            // Dodawanie błędów JSR-303 do BindingResult, aby Spring mógł je wyświetlić w widoku za pomocą obiektów FieldError
            result.addError(new FieldError("member", propertyPath, "Błąd: " + propertyPath + "(" + message + ")"));
        }

        // Można zastosować instrukcję violations.size() == 0 JSR-303 lub 
		// standardowe polecenie result.hasErrors() Springa, które teraz wyświetli błędy JSR-303
        if (!result.hasErrors()) {
            // Brak błędów
            memberService.add(member);
            model.addAttribute("guests", memberService.list());

            // Zauważ, że metoda SessionStatus.setComplete(); NIGDY nie jest wywoływana. 
            // Dzięki temu lista memberList jest wydłużana, ponieważ jest przechowywana w sesji.
            // Wywołanie SessionStatus.setComplete() spowodowałoby usunięcie wartości z listy memberList.
            // Zwracany widok to memberList. Jest on wiązany z plikiem
            // /WEB-INF/jsp/memberList.jsp
            return "memberList";
        } else {
            // Zwracanie widoku memberList, dzięki czemu użytkownik może poprawić błędy. Ten widok jest
            // wiązany z plikiem /WEB-INF/jsp/memberList.jsp
            return "memberList";
        }
    }

    // Metoda powiązana z adresem URL /member/remove
    @RequestMapping("remove")
    public String removeMember(@RequestParam("memberName")
    String memberName) {
        memberService.remove(memberName);

        // Wartość redirect powoduje odświeżenie listy.
        // Ponieważ kontroler używa symbolu wieloznacznego, można zastosować dowolny adres URL.
        // Tu podawany jest katalog główny ':' (czyli /member/)
        return "redirect:";
    }
}
