package org.springframework.prospring.ticket.web;

import java.util.*;

import javax.servlet.http.*;
import org.springframework.web.servlet.mvc.*;
import org.springframework.web.servlet.*;
import org.springframework.web.util.*;
import org.springframework.prospring.ticket.service.*;
import org.springframework.prospring.ticket.domain.*;

/**
 * Kontroler prezentujcy rezerwacj po okreleniu przez uytkownika iloci miejsc.
 * Oglnie rzecz biorc, ta strona mogaby by wywietlana przez kontroler BookSeetController, jednak
 * w takim przypadku gdyby uytkownik odwiey stron (na przykad, naciskajc klawisz F5), to formularz
 * zostaby ponownie przesany, a my musielibymy poprawnie obsugiwa t sytuacj w metodzie onSubmit() kontrolera.
 * Aby zaoszczdzi sobie tego problemu, przekierowujemy danie do tego kontrolera. Dziki temu, gdy uytkownik
 * odwiey stron, nowe danie zostanie obsuone przez ten kontroler, a nie przez kontroler formularza.
 */
public class ShowReservationController extends AbstractController {

    private String viewName;

    /**
     * @see AbstractController#handleRequestInternal(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    protected ModelAndView handleRequestInternal(HttpServletRequest request, HttpServletResponse response) throws Exception {

        Reservation reservation = getReservationFromSession(request);
        ReservationRequest reservationRequest = getReservationRequestFromSession(request);


        ModelAndView mav = new ModelAndView(viewName);
        mav.addObject("reservation", reservation);
        mav.addObject("performance", reservationRequest.getPerformance());
        mav.addObject("minutesReservationWillBeValid", new Integer(reservationRequest.getMinutesToHold()));
        mav.addObject("bookingFee", reservationRequest.getBookingFee());
        mav.addObject("seatsAdjacent", ((areAdjacentSeats(reservation.getSeats())) ? Boolean.TRUE : Boolean.FALSE));
        mav.addObject("seatsCount", new Integer(reservation.getSeats().length));
        return mav;

    }

    /**
     * Ustawia nazw widoku zwracanego przez ten kontroler.
     * @param viewName Nazwa widoku zwracanego przez ten kontroler.
     */
    public void setViewName(String viewName) {
        this.viewName = viewName;
    }

    // pobiera rezerwacj z sesji.
    protected Reservation getReservationFromSession(HttpServletRequest request) {
        return (Reservation)WebUtils.getRequiredSessionAttribute(request, "reservation");
    }

    // pobiera danie rezerwacji z sesji.
    protected ReservationRequest getReservationRequestFromSession(HttpServletRequest request) {
        return (ReservationRequest)WebUtils.getRequiredSessionAttribute(request, "reservationRequest");
    }

    /*
     * Sprawdza czy podane miejsca ssiaduj ze sob.
     *
     * algorytm:  Oglnie rzecz biorc, jeli wszystkie miejsca maj ze sob ssiadowa, to moe by
     *            tylko jedno miejsce, takie e miejsce z jego lewej stroni nie bdzie na licie
     *            oraz tylko jedno miejsce takie e miejsce z jego prawej strony nie bdzie na licie.
     *            Dla wszystkich pozostaych miejsc, miejsca z ich prawej oraz lewej strony 
     *            powinny by na licie.
     *
     *            Musimy przejrze wszystkie miejsca i dla kadego z nich sprawdzi czy 
     *            miejsca z jego lewej oraz prawej strony znajduj si na licie. Jeli
     *            odszukamy wicej ni jedno miejsce dla ktrego miejsce z jego lewej lub 
     *            prawej strony nie znajduje si na licie, to bdziemy wiedzie, e 
     *            miejsca nie s ssiadujce.
     */
    protected boolean areAdjacentSeats(Seat[] seatList) {

        boolean foundLeftNotInList = false;
        boolean foundRightNotInList = false;

        // umieszczamy wszystkie miejsca w obiekcie HashSet by zapewni szybszy dostp do nich.
        HashSet seats = new HashSet();
        for (int i=0; i<seatList.length; i++) {
            seats.add(seatList[i]);
        }

        for (int i=0; i<seatList.length; i++) {
            Seat seat = seatList[i];
            if (!seats.contains(seat.getLeftSide())) {
                if (foundLeftNotInList) {
                    return false;
                }
                foundLeftNotInList = true;
            }
            if (!seats.contains(seat.getRightSide())) {
                if (foundRightNotInList) {
                    return false;
                }
                foundRightNotInList = true;
            }
        }

        return true;
    }

}
