package org.springframework.prospring.ticket.web;

import java.beans.*;

import org.springframework.prospring.ticket.service.*;
import org.springframework.prospring.ticket.domain.*;

/**
 * Obiekt PropertyEditor (edytor waciwoci) dla klasy Reservation.
 *
 * Zazwyczaj definiowaby edytory waciwoci dla trwaych obiektw, ktre 
 * wczytywayby je na podstawie ich identyfikatorw. W tym przypadku, Reservation
 * nie jest trwaym obiektem, nie posiada identyfikatora i nie mone by wczytany 
 * z bazy danych. Zamiast tego edytor tworzy obiekt rezerwaqcji na podstawie 
 * informacji szczegowych, ktre s trwae.
 * Edytor operuje na identyfikatorze szczegowych informacji o rezerwacji.
 */
public class ReservationEditor extends PropertyEditorSupport {

    private BoxOffice boxOffice;

    public ReservationEditor(BoxOffice boxOffice) {
        this.boxOffice = boxOffice;
    }

    /**
     * Ustawia warto tego edytora.
     * @param bookingIdAsText acuch znakw reprezentujcy identyfikator informacji szczegowych 
     *        na podstawie ktrych zostanie utworzona rezerwacja.
     */
    public void setAsText(String bookingIdAsText) throws IllegalArgumentException {
        long bookingId = Long.valueOf(bookingIdAsText).longValue();

        // dwie ponisze instrukcje mona by zoptymalizowa do postaci tylko jednego wywoania
        // metody usugi BoxOffice, lecz zostawilimy je w takiej postaci, gdy dobrze
        // speniaj swoje zadanie.
        Booking booking = boxOffice.getBooking(bookingId);
        Seat[] seats = boxOffice.getBookedSeats(booking);

        // utworzenie rezerwacji i ustawienie jej jako wartoci edytora.
        setValue(new Reservation(seats, booking));
    }

    /**
     * Zwraca acuch znakw reprezentujcy obiekt Reservation stanowicy warto tego edytora.
     * Zwrcona warto jest w rzeczywistoci acuchow reprezentacj identyfikatora 
     * informacji szczegowych (wartoci tego edytora) na jakich bazuje rezerwacja.
     *
     * @see #setAsText(String)
     * @return Zwraca acuch znakw reprezentujcy obiekt Reservation stanowicy warto tego edytora.
     */
    public String getAsText() {
        Reservation reservation = (Reservation)getValue();
        return String.valueOf(reservation.getBooking().getId());
    }

}
