package org.springframework.prospring.ticket.service;

import java.math.*;
import java.util.*;

import org.springframework.prospring.ticket.domain.*;

/**
 * Reprezentuje danie rezerwacji. Jest to danie dotyczce zarezerwowania podanej iloci 
 * miejsc na konkretny spekatkl.
 */
public class ReservationRequest {

    private static final long MILLIS_PER_MINUTE = 1000L * 60L;

    // spektakl.
    private Performance performance;

    // ilo danych miejsc.
    private int numberOfSeatsRequested;

    // grupa cenowa danych miejsc.
    private PriceBand priceBand;

    // data okrelajca do kiedy miejsca bd zarezerwowane w ramach danej rezerwacji.
    private Date holdUntil;

    // ilo minut okrelajc jak dugo rezerwacja bdzie wana.
    private int minutesToHold;

    // opata za obsug rezerwacji.
    private BigDecimal bookingFee = new BigDecimal(0);

    /**
     *          
     */
    public ReservationRequest() {
    }

    /**
     * Konstruktor - tworzy nowe danie rezerwacji.
     * @param performance Spektakl, ktrego dotyczy rezerwacja.
     * @param priceBand Grupa cenowa miejsc.
     * @param numberOfSeatsRequested Ilo rezerwowanych miejsc.
     * @param minutesToHold Ilo minut na jakie miejsca zostan zarezerwowane.
     */
    public ReservationRequest(
        Performance performance,
        PriceBand priceBand,
        int numberOfSeatsRequested,
        int minutesToHold) {

        this.performance = performance;
        this.priceBand = priceBand;
        this.numberOfSeatsRequested = numberOfSeatsRequested;
        holdFor(minutesToHold);
    }

    /**
     * Zwraca dat okrelajca do kiedy miejsca bd zarezerwowane jeli nie zostanie wykonane zamwienie.
     * @return Data okrelajca do kiedy miejsca bd zarezerwowane jeli nie zostanie dokonane zamwienie.
     */
    public Date getHoldUntil() {
        return holdUntil;
    }

    /**
     * Zwraca ilo minut okrelajca jak dugo miejsca bd zarezerwowane jeli nie zostanie wykonane zamwienie.
     * @return Ilo minut okrelajca jak dugo miejsca bd zarezerwowane jeli nie zostanie wykonane zamwienie.
     */
    public int getMinutesToHold() {
        return minutesToHold;
    }

    /**
     * Ustawia ilo minut okrelajc jak dugo miejsca bd zarezerwowane jeli nie zostanie wykonane zamwienie.
     * @param minutes Ilo minut okrelajca jak dugo miejsca bd zarezerwowane jeli nie zostanie wykonane zamwienie.
     */
    public void holdFor(int minutes) {
        minutesToHold = minutes;
        holdUntil = new Date(System.currentTimeMillis() + minutes * MILLIS_PER_MINUTE);
    }

    /**
     * Zwraca spektakl skojarzony z tym daniem.
     * @return Spektakl skojarzony z tym daniem.
     */
    public Performance getPerformance() {
        return performance;
    }

    /**
     * Ustawia spektakl skojarzony z tym daniem.
     * @param performance Spektakl skojarzony z tym daniem.
     */
    public void setPerformance(Performance performance) {
        this.performance = performance;
    }

    /**
     * Zwraca ilo miejsc skojarzon z tym daniem.
     * @return Ilo miejsc skojarzona z tym daniem.
     */
    public int getNumberOfSeatsRequested() {
        return numberOfSeatsRequested;
    }

    /**
     * Ustawia ilo miejsc skojarzon z tym daniem.
     * @param numberOfSeatsRequested Ilo miejsc skojarzona z tym daniem.
     */
    public void setNumberOfSeatsRequested(int numberOfSeatsRequested) {
        this.numberOfSeatsRequested = numberOfSeatsRequested;
    }

    /**
     * Zwraca grup cenow rezerwowanych miejsc.
     * @return Grupa cenowa rezerwowanych miejsc.
     */
    public PriceBand getPriceBand() {
        return priceBand;
    }

    /**
     * Ustawia grup cenow rezerwowanych miejsc.
     * @param priceBand Grupa cenowa rezerwowanych miejsc.
     */
    public void setPriceBand(PriceBand priceBand) {
        this.priceBand = priceBand;
    }

    /**
     * Zwraca opat za obsug rezerwacji.
     * @return Opata za obsug rezerwacji.
     */
    public BigDecimal getBookingFee() {
        return bookingFee;
    }

    /**
     * Ustawia opat za obsug rezerwacji.
     * @param bookingFee Opata za obsug rezerwacji.
     */
    public void setBookingFee(BigDecimal bookingFee) {
        this.bookingFee = bookingFee;
    }

}
