package org.springframework.prospring.ticket.dao;

import java.math.*;
import java.util.*;

import org.springframework.prospring.ticket.domain.*;
import org.springframework.prospring.ticket.domain.support.*;
import org.springframework.prospring.ticket.service.*;


/**
 * Interfejs definiujcy wszystkie metody dostpu do danych w naszej przykadowej aplikacji
 */
public interface BoxOfficeDao {

    /**
     * Oznacza wskazane miejsca jako zarezerwowane na konkretny spektakl i skojarzone z podan rezerwacj.
     * @param seats Miejsca jakie naley oznaczy jako zarezerwowane.
     * @param performance Spektal na jaki miejsca zostan zarezerwowane.
     * @param booking Rezerwacja z jak miejsca s skojarzone.
     * @throws RequestedSeatNotAvailableException Zgaszany jeli ktrekolwiek z miejsc nie jest ju dostpne,
     *         czyli jeli wczeniej zostao zarezerwowane.
     */
    public void reserveSeats(Seat[] seats, Performance performance, Booking booking) 
			throws RequestedSeatNotAvailableException;

    /**
     * Zapisuje rezerwacj w bazie danych.
     * @param booking Rezerwacja jak naley zapisa w bazie danych.
     */
    public void savePurchaseForBooking(Booking booking);

    /**
     * Zwraca wszystkie rodzaje przedstawie zapisane w bazie.
     * @return Wszystkie rodzaje przedstawie zapisane w bazie.
     */
    public List getAllGenres();

    /**
     * Returns all generes with shows in the database
     * Zwraca wszystkie rodzaje przedstawie dla jakich w bazie s jakie spektakle.
     * @return Wszystkie rodzaje przedstawie dla jakich w bazie s jakie spektakle.
     */
    public List getCurrentGenres();

    /**
     * Zwraca przedstawienie skojarzone z podanym identyfikatorem przedstawienia.
     * @param id Identyfikator przedstawienia.
     * @return Przedstawienie skojarzone z podanym identyfikatorem.
     */
    public Show getShow(long id);

    /**
     * Zwraca spektakl skojarzony z podanym identyfikatorem spektaklu.
     * @param id Identyfikator spektaklu.
     * @return Spektakl skojarzony z podanym identyfikatorem.
     */
	public Performance getPerformance(long id);

    /**
     * Zwraca zakres cen skojarzony z podanym identyfiaktorem cen.
     * @param id Identyfikator zakresu cen.
     * @return Zakres cen skojarzony z podanym identyfikatorem.
     */
    public PriceBand getPriceBand(long id);

    /**
     * Zwraca struktur cen skojarzon z podanym identyfikatorem struktury cen.
     * @param id Identyfikator struktury cen.
     * @return Struktura cen skojarzona z podanym identyfikatorem.
     */
    public PriceStructure getPriceStructure(long id);

    /**
     * Zwraca list wszystkie spektakli podanego przedstawienia. Zwracane spektakle zawieraj informacje
     * o dostpnoci miejsc.
     * @param show Przedstawienie.
     * @return Lista wszystkie spektakli (wraz z informacjami o dostpnych miejscach) danego przedstawienia.
     */
    public PerformanceWithAvailability[] getAvailabilityForPerformances(Show show);

    /**
     * Zwraca list wszystkich dostpnych miejsc podanek kategorii na dany spektakl.
     * @param performance Spektakl.
     * @param seatClass Kategoria miejsc.
     * @return Lista wszystkich dostpnych miejsc podanej kategorii na dany spektakl.
     */
    public Seat[] getAvailableSeats(Performance performance, SeatClass seatClass);

    /**
     * Zwraca ilo dostpnych miejsc podanej kategorii na wskazany spektakl.
     * @param performance Spektakl.
     * @param seatClass Kategoria miejsc.
     * @return Zwraca ilo dostpnych miejsc podanej kategorii na wskazany spektakl.
     */
    public int getAvailableSeatsCount(Performance performance, SeatClass seatClass);

    /**
     * Zwraca koszt podanych miejsc na wskazany spektakl.
     * @param performance Spektakl.
     * @param seats Miejsca.
     * @return Koszt podanych miejsc na wskazany spektakl.
     */
    public BigDecimal getCostOfSeats(Performance performance, Seat[] seats);

    /**
     * Usuwa rezerwacj z bazy danych.
     * @param booking Rezerwacja jak naley usun z bazy danych.
     */
    public void removeBooking(Booking booking);

    /**
     * Zwraca rezerwacj skojarzon z podanym identyfikatorem.
     * @param bookingId Identyfikator rezerwacji.
     * @return Rezerwacja skojarzona z podanym identyfiaktorem.
     */
    public Booking getBooking(long bookingId);

    /**
     * Zwraca miejsca skojarzone z dan rezerwacj.
     * @param booking Rezerwacja.
     * @return Miejsca skojarzone z podan rezerwacj.
     */
    public List getSeatsForBooking(Booking booking);

    /**
     * Zwraca spektakl skojarzony z podan rezerwacj.
     * @param booking Rezerwacja.
     * @return Spektakl skojarzony z podan rezerwacj.
     */
    public Performance getPerformanceForBooking(Booking booking);

    /**
     * Sprawdza dostpno miejsc skojarzonych z podan rezerwacj.
     * @param booking Sprawdzana rezerwacja.
     * @return True jeli miejsca s dostpne, false w przeciwnym przypadku.
     */
    public boolean checkAvailabilityForBooking(Booking booking);

    /**
     * Aktualizuje kod uwierzytelniajcy danego zakupu w bazie danych. 
     * @param purchase Aktualizowana rezerwacja.
     */
    public void updatePurchaseAuthorizationCode(Purchase purchase);
}
