package org.springframework.prospring.ticket.web.confirmation;

import org.springframework.prospring.ticket.domain.*;
import org.springframework.prospring.ticket.service.*;
import org.springframework.mail.*;

/**
 * Wysya prost wiadomo poczty elektronicznej z potwierdzeniem zamwienia.
 */
public class SimpleMailPurchaseConfirmationSender implements PurchaseConfirmationSender {

    // usuga uywana do wysyania wiadomoci poczty elektronicznej.
    private MailSender mailSender;

    // adres poczty elektronicznej nadawcy wiadomoci z potwierdzeniem.
    private String senderEmail;

    public void sendPurchaseConfirmation(Purchase purchase, PurchaseRequest purchaseRequest)
        throws PurchaseConfirmationSendingException {

        try {
            mailSender.send(createPurchaseConfirmationMessage(purchase, purchaseRequest));
        } catch (MailException me) {
            throw new PurchaseConfirmationSendingException("Nie mona byo wysa wiadomoci z potwierdzeniem zamwienia nr. " +
                purchase.getId(), me);
        }
    }

    /**
     * Ustawia adres nadawcy wysyajcego wiadomoci z potwierdzeniem.
     * @param mailSender Adres nadawcy wysyajcego wiadomoci z potwierdzeniem.
     */
    public void setMailSender(MailSender mailSender) {
        this.mailSender = mailSender;
    }

    /**
     * Sets the email that will apear in the "from" field of the email messages.
     * Ustawia adres poczty elektronicznej, ktry bdize widoczny w polu "from" (od) potwierdzenia.
     * @param senderEmail Adres poczty elektronicznej, ktry bdize widoczny w polu "from" (od) potwierdzenia.
     */
    public void setSenderEmail(String senderEmail) {
        this.senderEmail = senderEmail;
    }

    /**
     * Tworzy wiadomo z potwierdzeniem danego zamwienia. Ta metoda moe zosta przesonita
     * w celu generacji bardziej zoonych wiadomoci (na przykad, przy wykorzystaniu szablonw
     * Velocity).
     *
     * @param purchase Zamwienie ktre potwierdzamy.
     * @param request danie zamwienia.
     * @return Wiadomo stanowica potwierdzenie danego zamwienia.
     */
    protected SimpleMailMessage createPurchaseConfirmationMessage(Purchase purchase, PurchaseRequest request) {
        SimpleMailMessage message = new SimpleMailMessage();
        message.setTo(purchase.getEmail());
        message.setFrom(senderEmail);
        message.setSubject("Potwierdzenie zamwienia " + purchase.getPaymentAuthorizationCode());
        message.setText(createMessageText(purchase, request));
        return message;
    }

    protected String createMessageText(Purchase purchase, PurchaseRequest request) {
        Performance performance = request.getPerformance();

        StringBuffer text = new StringBuffer();
        text.append("Zoone przez Ciebie zamwienie miejsc na spektakl ").append(performance.getShow().getName());
        text.append(" dnia ").append(performance.getDateAndTime()).append(" zostao przyjte.\n\n");
        text.append("Twoja karta kredytowa o numerze ").append(purchase.getEncryptedCardNumber()).append(" zostanie obciona kwot ");
        text.append(request.getReservation().getBooking().getPrice()).append("PLN.\n\n");
        text.append("Zostay Ci przydzielone miejsca ");
        Seat[] seats = request.getReservation().getSeats();
        for (int i=0; i<seats.length; i++) {
            if (i > 0) {
                text.append("; ");
            }
            text.append(seats[i].getName());
        }
        text.append(".\n\n");
        if (purchase.isUserCollected()) {
            text.append("Twoje bilety czekaj na odbir w kasie teatru.");
        } else {
            text.append("Twoje bilety zostan przesane poczt na nastpujcy adres:\n\n");
            Address deliveryAddress = getDeliveryAddress(purchase);
            text.append(purchase.getCustomerName()).append("\n");
            text.append(deliveryAddress.getStreet()).append("\n");
            text.append(deliveryAddress.getCity()).append("\n");
            text.append(deliveryAddress.getCountry()).append("\n");
            text.append(deliveryAddress.getPostcode());
        }
        text.append("\n\n");
        text.append("Numer referencyjny zamwienia: ").append(purchase.getPaymentAuthorizationCode()).append(".");
        text.append("Prosimy o zachowanie tego numeru (na przykad wydrukuj t wiadomo). W przypadku wtpliwoci ");
        text.append("lub anulowania rezerwacji bdziesz go potrzebowa.\n\n");
        return text.toString();
    }

    private Address getDeliveryAddress(Purchase purchase) {
        if (purchase.getDeliveryAddress() == null) {
            return purchase.getBillingAddress();
        }
        Address deliveryAddress = purchase.getDeliveryAddress();
        if (deliveryAddress.getCountry() == null || deliveryAddress.getCountry().length() == 0) {
            return purchase.getBillingAddress();
        }
        return deliveryAddress;
    }
}
