package org.springframework.prospring.ticket.web;

import java.text.*;
import java.util.*;

import javax.servlet.http.*;

import org.apache.commons.logging.*;
import org.springframework.beans.propertyeditors.*;
import org.springframework.prospring.ticket.domain.*;
import org.springframework.prospring.ticket.service.*;
import org.springframework.prospring.ticket.service.payment.*;
import org.springframework.prospring.ticket.web.confirmation.*;
import org.springframework.validation.*;
import org.springframework.web.bind.*;
import org.springframework.web.servlet.*;
import org.springframework.web.servlet.mvc.*;
import org.springframework.web.util.*;

/**
 * Kontroler formularza "Szczegy patono".
 *
 * @author Uri Boness
 */
public class PaymentFormController extends SimpleFormController {

    private final static Log logger = LogFactory.getLog(PaymentFormController.class);

    // Format zapisu daty wanoci karty kredytowej.
    private final static DateFormat DATE_FORMAT = new SimpleDateFormat("MM/yy");

    // Usuga BoxOffice uywana przez ten kontroler.
    private BoxOffice boxOffice;

    // nadawca jaki zostanie podany w potwierdzeniach wysyanych poczt elektroniczn.
    private PurchaseConfirmationSender confirmationSender;

    // nazwa widoku wywietlanego w sytuacji gdy kto inny wczeniej zarezerwowa wybrane miejsca.
    private String seatsGoneViewName;

    /**
     * Rejestracja edytorw waciwoci dla pl formularza.
     * @see BaseCommandController#initBinder(
     *      javax.servlet.http.HttpServletRequest,
     *      org.springframework.web.bind.ServletRequestDataBinder)
     */
    protected void initBinder(HttpServletRequest request, ServletRequestDataBinder binder) throws Exception {
        binder.registerCustomEditor(Date.class, new CustomDateEditor(DATE_FORMAT, false));
        binder.registerCustomEditor(Reservation.class, new ReservationEditor(boxOffice));
    }

    /**
     * Tworzy obiekt PurchaseRequest sucy jako obiekt wsppracujcy z formularzem.
     * @see AbstractFormController#formBackingObject(javax.servlet.http.HttpServletRequest)
     */
    protected Object formBackingObject(HttpServletRequest request) throws Exception {
        Reservation reservation = loadReservation(request);
        Performance performance = loadReservationRequest(request).getPerformance();
        return createPurchaseRequest(reservation, performance);
    }

    /**
     * Sprawdzenie czy miejsca skojarzone z t rezerwacj wci s dostpne. Jeli nie s, to uytkownik 
     * zostanie przekierowany do widoku, ktrego nazw jest okrelona przez waciwo seatsGoneViewName.
     * redirecting to "seatsGone" view.
     */
    protected ModelAndView showForm(HttpServletRequest request, HttpServletResponse response, BindException errors) throws Exception {
        Reservation reservation = loadReservation(request);
        Performance performance = loadReservationRequest(request).getPerformance();
        if (!boxOffice.areSeatsStillAvailable(reservation.getBooking())) {
            ModelAndView mav = new ModelAndView(seatsGoneViewName);
            mav.addObject("performance", performance);
            return mav;
        }
        return super.showForm(request, response, errors);
    }

    /**
     * Wywoywanan w momencie przesania formularza (po utworzenie obiektu wsppracujcego z formularzem,
     * jego zainicjalizowaniu (przez obiekt wicy), powizaniu i zweryfikowaniu poprawnoci).
     *
     * @see SimpleFormController#onSubmit(
     *      javax.servlet.http.HttpServletRequest,
     *      javax.servlet.http.HttpServletResponse,
     *      Object,
     *      org.springframework.validation.BindException)
     */
    protected ModelAndView onSubmit(
        HttpServletRequest request,
        HttpServletResponse response,
        Object command,
        BindException errors) throws Exception, RequestedSeatNotAvailableException {

        PurchaseRequest purchaseRequest = (PurchaseRequest)command;

        Reservation reservation = getAndRemoveReservationFromSession(request);

        // jeli rezerwacja jest rwna null, to bdzie to oznacza, e formularz zosta ponownie
        // przesany, w takim przypadku nie bdziemy chcieli ponownie przetwarza zamwienia.
        if (reservation == null) {
            return handleResubmission(purchaseRequest);
        }

        // przetworzenie zamwienia.
        Purchase purchase;
        try {

            purchase = boxOffice.purchaseTickets(purchaseRequest);

        } catch (CreditCardValidationException ccve) {
            keepReservation(request, reservation); // umieszczenie rezerwacji w sesji.
            errors.reject(ccve.getErrorCode());
            return showForm(request, errors, getFormView());
        } catch (PaymentException pe) {
            keepReservation(request, reservation); // umieszczenie rezerwacji w sesji.
            errors.reject(pe.getErrorCode());
            return showForm(request, errors, getFormView());
        }

        // wysanie potwierdzenia poczt elektroniczn.
        try {
            confirmationSender.sendPurchaseConfirmation(purchase, purchaseRequest);
        } catch (PurchaseConfirmationSendingException pcse) {
            logger.error("Nie udao si wysa wiadomoci potwierdzajcej zamwienie nr. " + purchase.getId(), pcse);
        }

        ModelAndView mav = new ModelAndView(getSuccessView());
        mav.addObject("useBillingAddressForDelivery", shouldUseBillingAddressForDelivery(purchase));
        mav.addObject("reservation", purchaseRequest.getReservation());
        mav.addObject("performance", purchaseRequest.getPerformance());
        mav.addObject("purchase", purchase);
        return mav;
    }

    /**
     * Obsuguje powtrne przesanie formularza do rozlicze finansowych.
     * @param purchaseRequest danie zakupu.
     * @return Model i widok uywany do wywietlenia informacji i zakupie.
     */
    protected ModelAndView handleResubmission(PurchaseRequest purchaseRequest) {
        Purchase purchase = purchaseRequest.getReservation().getBooking().getPurchase();
        ModelAndView mav = new ModelAndView(getSuccessView());
        mav.addObject("useBillingAddressForDelivery", shouldUseBillingAddressForDelivery(purchase));
        mav.addObject("reservation", purchaseRequest.getReservation());
        mav.addObject("performance", purchaseRequest.getPerformance());
        mav.addObject("purchase", purchase);

        return mav;
    }

    /**
     * Ustawia usug BoxOffice uywan przez ten kontroler.
     * @param boxOffice Usuga BoxOffice uywana przez ten kontroler.
     */
    public void setBoxOffice(BoxOffice boxOffice) {
        this.boxOffice = boxOffice;
    }

    /**
     * Ustawia usug wysyajc potwirdzenia zakupy uywan przez ten kontroler.
     * @param confirmationSender Usuga wysyajca potwirdzenia zakupy uywan przez ten kontroler.
     */
    public void setConfirmationSender(PurchaseConfirmationSender confirmationSender) {
        this.confirmationSender = confirmationSender;
    }

    /**
     * Ustawia nazw widoku wywietlanego gdy przydzielone miejsca zostay wczeniej zarezerwowane przez kogo innego.
     * @param seatsGoneViewName Nazwa widoku informujcego o tym, e miejsca zostay zarezerwowane.
     */
    public void setSeatsGoneViewName(String seatsGoneViewName) {
        this.seatsGoneViewName = seatsGoneViewName;
    }

    //========================================= Metody pomocnicze ================================================

    /**
     * Okrela czy jako adresu wysyki naley uy adresu do rozlicze finansowych.
     * @param purchase Obiekt reprezentujcy zakup.
     * @return True jeli adres do rozlicze ma zosta uyty jako adres wysyki, false w przeciwnym razie.
     */
    protected Boolean shouldUseBillingAddressForDelivery(Purchase purchase) {
        Address address = purchase.getDeliveryAddress();
        if (address == null) {
            return Boolean.TRUE;
        }
        return (address.getCountry() == null || address.getCountry().length() == 0)? Boolean.TRUE : Boolean.FALSE;
    }

    // wczytuje rezerwacj z dania.
    protected Reservation loadReservation(HttpServletRequest request) {
        return (Reservation)WebUtils.getRequiredSessionAttribute(request, "reservation");
    }

    // wczytuje spektakl z dania.
    protected ReservationRequest loadReservationRequest(HttpServletRequest request) {
        return (ReservationRequest)WebUtils.getRequiredSessionAttribute(request, "reservationRequest");
    }

    // umieszcza przekazan rezerwacj w sesji.
    protected void keepReservation(HttpServletRequest request, Reservation reservation) {
        WebUtils.setSessionAttribute(request, "reservation", reservation);
    }

    // umieszcza przekazany spektakl w sesji.
    protected void keepPerformance(HttpServletRequest request, Performance performance) {
        WebUtils.setSessionAttribute(request, "performance", performance);
    }

    // tworzy nowe danie zakupu na podstawie podanej rezerwacji.
    protected PurchaseRequest createPurchaseRequest(Reservation reservation, Performance performance) {
        PurchaseRequest request = new PurchaseRequest();
        request.setBillingAddress(new Address());
        request.setDeliveryAddress(new Address());
        request.setCreditCardDetails(new CreditCardDetails());
        request.setReservation(reservation);
        request.setPerformance(performance);
        return request;
    }

    // pobiera i usuwa rezerwacj z sesji. 
    protected Reservation getAndRemoveReservationFromSession(HttpServletRequest request) {
        Reservation reservation = (Reservation)WebUtils.getSessionAttribute(request, "reservation");
        request.getSession(true).removeAttribute("reservation");
        return reservation;
    }

    // pobiera i usuwa danie rezerwacji z sesji.
    protected ReservationRequest getAndRemoveReservationRequestFromSession(HttpServletRequest request) {
        ReservationRequest reservationRequest = (ReservationRequest)WebUtils.getSessionAttribute(request, "reservationRequest");
        request.getSession(true).removeAttribute("reservationRequest");
        return reservationRequest;
    }

}
